/***************************************************************************
								   memmgr.c  
							 -------------------
	Description 		  C code for memory manager
 ***************************************************************************/
/***************************************************************************
 *																		   *
 *	 This program is free software; you can redistribute it and/or modify  *
 *	 it under the terms of the GNU General Public License as published by  *
 *	 the Free Software Foundation; either version 2 of the License, or	   *
 *	 (at your option) any later version.								   *
 *																		   *
 ***************************************************************************/

#include "config.h"

#include <assert.h>
#include <malloc.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h> 

#include "memmgr.h"
#include "tlisp.h"

// Low level stuff

#define BLKREQ 4096  //number of MemBlocks to request from OS each time

memMgrT	g_memmgr = {BLKREQ};

Lptr assign(Lptr Sc) 
{
	Rcount(Sc)++;
	return Sc;
}

Lptr decrement(Lptr Sc) 
{
	Rcount(Sc)--;
	return Sc;
}

void release(Lptr Snode)
{
	if (--(Rcount(Snode)) <= 0) {
		if (Vempty(Snode)) {
			if (Rcount(Snode)) {
				fprintf(stderr, "Release invoked on %s without prior assign\n", Vname(Snode));
			}
			freeMemBlock((freeT *) Snode);
		}
}	}

void setSptr(Lptr *sptrP, Lptr sC)
{
	Lptr old = *sptrP;

	if (sC) {
		assign(sC);
	}
	*sptrP = sC;
	if (old != NULL) {
		release(old);
}	}

// Start of single instantiations

static int cons_sizeof(void)
{
	return(sizeof(consT));
}

static celltypeE nil_stype(void)
{
	return(NILTYPE);
}

static const char * nil_name(void)
{
	return("NIL");
}

static void nil_dump(void *P, FILE *F)
{
	fprintf(F, "NIL");
}

static int	nil_empty(void *Snode)
{
	Rcount((Mptr) Snode) = 0x7FFFFFFF;
	return(0);
}

static void nil_set_mark(Lptr Snode)
{
	fprintf(stderr, "Can't set mark in %s\n", Vname(Snode));
}

static void nil_clr_mark(Lptr Snode)
{
}

static vtableT nil_vtable =
{
	&cons_sizeof,
	NILTYPE,
	&nil_name,
	&nil_dump,
	&nil_empty,
	&nil_set_mark,
	&nil_clr_mark,
	&is_false
};

static MemBlock nilScell =	{&nil_vtable, 0x7FFFFFFF, 0, &nilScell, &nilScell };

Lptr getNil(void)
{
	return &nilScell;		 
}

static celltypeE undef_stype(void)
{
	return(UNDEFTYPE);
}

static const char * undef_name(void)
{
	return("*UNDEF*");
}

static void undef_dump(void *P, FILE *F)
{
	fprintf(F, "*UNDEF*");
}

static vtableT undef_vtable =
{
	&cons_sizeof,
	UNDEFTYPE,
	&undef_name,
	&undef_dump,
	&nil_empty,
	&nil_set_mark,
	&nil_clr_mark,
	&is_false
};

Lptr getUndef(void)
{
	static MemBlock undefScell = {&undef_vtable, 0x7FFFFFFF, 0, &undefScell, &undefScell };

	return &undefScell; 	   
}

static celltypeE error_stype(void)
{
	return(ERRORTYPE);
}

static const char * error_name(void)
{
	return("*ERROR*");
}

static void error_dump(void *P, FILE *F)
{
	fprintf(F, "*ERROR*");
}

static vtableT error_vtable =
{
	&cons_sizeof,
	ERRORTYPE,
	&error_name,
	&error_dump,
	&nil_empty,
	&nil_set_mark,
	&nil_clr_mark,
	&is_false
};

Lptr getError(void)
{
	static MemBlock errorScell = {&error_vtable, 0x7FFFFFFF, 0, &errorScell, &errorScell };

	return &errorScell;
}

// ----------------------------
// Start of versioning software
// ----------------------------

Lptr PopLog = &nilScell;
static int currentVer = 0;

/*	VERSION PER NODE FUNCTIONS */
int getVersion(Lptr S2node) {
	return (Version(S2node));
}

int getCurrentVersion(void) {
	return(currentVer);
}

static void setVersion(Lptr S2node,int newversion) {
	Version(S2node) = newversion;
}

/* Make a cons cell given a pair of car and
   cdr parameters
   ==========================================
   Parameters: incar, incdr are the car/cdr
			   parameters
   ==========================================
   Returns: the new cons cell
   ==========================================
   Note: the reference counts are incremented
		 by calling assign;
*/	

static celltypeE cons_stype(void)
{
	return(CONSTYPE);
}

static const char * cons_name(void)
{
	return("CONS");
}

static void cons_dump(void *P, FILE *F)
{
	fprintf(F, "%p Cons refcnt=%u", P, Rcount((Mptr) P));
}

static int	cons_empty(void *cons)
{
	release(Cdr((Mptr) cons));
	release(Car((Mptr) cons));
	return(1);
}

static void cons_set_mark(Lptr fnode);
static void cons_clr_mark(Lptr fnode);

static vtableT cons_vtable =
{
	&cons_sizeof,
	CONSTYPE,
	&cons_name,
	&cons_dump,
	&cons_empty,
	&cons_set_mark,
	&cons_clr_mark,
	&is_false
};

static vtableT cons_vtable_marked =
{
	&cons_sizeof,
	CONSTYPE,
	&cons_name,
	&cons_dump,
	&cons_empty,
	&cons_set_mark,
	&cons_clr_mark,
	&is_true
};

static void cons_set_mark(Lptr fnode)
{
	Vtable(fnode) = &cons_vtable_marked;
}

static void cons_clr_mark(Lptr fnode)
{
	Vtable(fnode) = &cons_vtable;
}

Lptr cons(Lptr incar, Lptr incdr)
{
	Lptr temp;
	
	temp            = (Lptr) getMemBlock(&cons_vtable);
	Rcount(temp)    = 0;
	Version(temp)	= 0;
	assign(incar);
	Car(temp)		= incar;
	assign(incdr);
	Cdr(temp)		= incdr;
	
	return temp;
}

/* cons two cells together to a cons cell and assign
   a version number */

static Lptr consVER(Lptr incar, Lptr incdr, int ver) {
	Lptr Retcons;
	Retcons = cons(incar, incdr);
	setVersion(Retcons, ver);
	return(Retcons);
}

/* Get the car node of the given cons cell
   ==========================================
   Parameters: snode
   ==========================================
   Returns: the car node of the parameter node
   ==========================================
   Note: _car does not consider version info
		 if it is not cons cell, returns nil
*/	 
static Lptr icar(Lptr Snode){
	if (Vtype(Snode) == CONSTYPE) {
		return ( Car(Snode) );
	}
	return (getNil());
}

/* Get the cdr node of the given cons cell
   ==========================================
   Parameters: snode
   ==========================================
   Returns: the cdr node of the parameter node
   ==========================================
   Note: _cdr does not consider version info;
		 if it is not a cons cell, returns nil
*/	 
static Lptr icdr(Lptr Snode){
	if (Vtype(Snode) == CONSTYPE){
		return ( Cdr(Snode) );
	}
	return (getNil());
}

/* Replaces the car of the cons cell passed in S2node
   with Sanode, and returns the new cons cell
   =====================================================
   Note: _rplaca considers no version info;
		 makes no change if S2node is not a cons cell;
		 _rplaca changes the reference counts by calling
		 assign and release.
*/
static Lptr irplaca(Lptr S2node, Lptr Sanode){
	
	if (Vtype(S2node) != CONSTYPE ) {
		assert(0);
	} else {
		setSptr(&(Car(S2node)), Sanode);
	} 
	return S2node;
}

/* Replaces the cdr of the cons cell passed in S2node
   with Sanode, and returns the new cons cell
   =====================================================
   Note: _rplacd considers no version info;
		 makes no change if S2node is not a cons cell;
		 _rplacd changes the reference counts by calling
		 assign and release.
*/
static Lptr irplacd(Lptr S2node, Lptr Sdnode){
	
	if (Vtype(S2node) != CONSTYPE ) {
		assert(0);
	} else {
		setSptr(&(Cdr(S2node)), Sdnode);
	} 
	return S2node;
}

/* PopLog structure:
 * PopLog is a pointer to a chain of (zero or more) header nodes using the
 * car with null indicated by the zero pointer.  The head of this chain
 * contains the header node for the current version, subsequent headers
 * count down to version 1.
 * Each header addresses in its cdr a chain of (zero or more) nodes each
 * chained to the next by the cdr pointer.
 * Each such node uses its car point to address a specific cons node whose
 * version is consistent with the version associated with the nodes in this
 * chain.
 */

void beginVer(void) {
	// Create a new header addressed by the PopLog ..
	// Any old header is on the chain of cars
	setSptr(&PopLog, consVER(PopLog, getNil(), ++currentVer));
}

static void addPopLog(Lptr master) {
	// Chain in as the cdr of Poplog a new node whose car points at master and cdr supports the chain

	assert(Vtype(master) == CONSTYPE);
	setVersion(master, currentVer);
	irplacd(PopLog, cons(master, icdr(PopLog)));
}

void commitVer(void)
{
	Lptr current, log, master, last;

	// Hold onto current head of PopLog
	current = assign(PopLog);

	// Now move back one version in the PopLog
	setSptr(&PopLog, icar(PopLog));
	--currentVer;

	// For each item on the cdr chain of the current PopLog
	// commit the prior version with whatever is current
	// while destructively releasing the nodes on this chain
	for (; ; irplacd(current, icdr(log)) ) {
		log = icdr(current);
		if (Vtype(log) != CONSTYPE) {
			break;
		}
        master = icar(log);

		assert(Vtype(master) == CONSTYPE);
		if (getVersion(master) != currentVer + 1) {
			// Can be less if used pputprop
			assert(getVersion(master) <= currentVer);
		} else {
			Lptr	real = assign(icar(master));
			setVersion(master, currentVer);
			if (!currentVer) {
				irplaca(master, car(real));
				irplacd(master, cdr(real));
			} else {
				last   = icdr(master);
				switch (Vtype(last)) {
				case NILTYPE:
					// No prior version
					break;
				case CONSTYPE:
					if (getVersion(last) == currentVer) {
						irplacd(master, icdr(last));
					} else {
						assert(getVersion(last) < currentVer);
						addPopLog(master);
					}
					break;
				default:
					assert(0);
			}	}
			release(real);
	}	}
	release(current);
}

void abortVer(void) {
	Lptr log, master, last;

	// For each item on the cdr chain of the current PopLog
	// unwind the version with whatever was the last
	// while destructively releasing the nodes on this chain
	for (; ; irplacd(PopLog, icdr(log)) ) {
		log = icdr(PopLog);
		if (Vtype(log) != CONSTYPE) {
			break;
		}
		master = icar(log);
		assert(Vtype(master) == CONSTYPE);
		assert(getVersion(master) == currentVer);
		last   = icdr(master);
		switch (Vtype(last)) {
		case NILTYPE:
			// Can discard this Cons node
			irplaca(master, getNil());
			break;
		case CONSTYPE:
			setVersion(master, getVersion(last));
			irplaca(master, icar(last));
			irplacd(master, icdr(last));
			break;
		default:
			assert(0);
	}	}
	// Now remove the thing addressed by PopLog
	setSptr(&PopLog, icar(PopLog));
	--currentVer;
}

void resetLogVer(void) {

	while (PopLog && Vtype(PopLog) == CONSTYPE) {
		abortVer();
	}
	assert(!currentVer);
}

Lptr car(Lptr Snode) {
	Lptr ret = icar(Snode);

	if (!getVersion(Snode)) {
		return(ret);
	}
	return(icar(ret));
}

Lptr cdr(Lptr Snode) {
	if (!getVersion(Snode)) {
		return(icdr(Snode));
	}
	return(icdr(icar(Snode)));
}

#include "lsedit.h"

static void
badVersion(Lptr S2node)
{
//	lseditT parms = { "..", 1, 1, 1};
	int 	version;
	
	version = getVersion(S2node);
	printf("Node %p has version %d > current version %d\n", S2node, version, currentVer);

//	dump_lsedit(&parms, S2node);
	assert(version < currentVer);
}
	
Lptr rplaca(Lptr S2node, Lptr Sanode) {
	int  version;
	
	if (Vtype(S2node) != CONSTYPE) {
		assert(0);
	} else {
		version = getVersion(S2node);

		if (version == currentVer) {
			if (currentVer) {
				irplaca(icar(S2node), Sanode);
			} else {
				irplaca(S2node, Sanode);
			}
		} else {
			Lptr OldMaster, newCData;

			if (version > currentVer) {
				badVersion(S2node);
			}

			// Create a duplicate of the current S2Node
			OldMaster = consVER(icar(S2node),icdr(S2node), version);

			// N.B Since version > 0 will always be (version cons)->(cons)
			newCData = cons(Sanode, cdr(S2node));
			irplacd(S2node, OldMaster);
			irplaca(S2node, newCData);
			
			// Add S2node to the pop log
			addPopLog(S2node);
	}	}
	return(S2node);
}

Lptr prplaca(Lptr S2node, Lptr Sanode)
{
	if (getVersion(S2node) != 0) {
		Lptr	real = car(S2node);

		irplacd(S2node, cdr(real));
		setVersion(S2node, 0);
	}
	irplaca(S2node, Sanode);
	return(S2node);
}

Lptr rplacd(Lptr S2node, Lptr Sdnode) {
	int version;
	
	if (Vtype(S2node) != CONSTYPE) {
		assert(0);
	} else {
		version = getVersion(S2node);

		if (version == currentVer) {
			if (currentVer) {
				irplacd(icar(S2node), Sdnode);
			} else {
				irplacd(S2node, Sdnode);
			}
		} else {
			Lptr OldMaster, newCData;

			if (version > currentVer) {
				badVersion(S2node);
			}

			// Create a duplicate of the current S2Node
			OldMaster = consVER(icar(S2node),icdr(S2node),getVersion(S2node));

			// N.B Since version > 0 will always be (version cons)->(cons)
			newCData = cons(car(S2node), Sdnode);
			irplacd(S2node, OldMaster);
			irplaca(S2node, newCData);

			// Add S2node to the pop log
			addPopLog(S2node);
	}	}
	return(S2node);
}

// ===================
// Start of hash logic
// ===================

/**********************************************************************************
  Hash Function
 **********************************************************************************/

/* The global hash table */
Lptr hashtable[MAXHASHSIZE];

/* A global indicator that whether the global hash table has been
   initialized */
static int initialized = 0;

/* Given a key, calculate its hash value
   ======================================
   Paramters:
	 key	  the key to be hashed
	 length   the length for the hash function
*/	 
static unsigned int hash(char *key, size_t length) {
	unsigned int hash, i, mask=(MAXHASHSIZE-1);
	
	for (hash=0, i=0; i<length; ++i) {
		hash += key[i];
		hash += (hash << 10);
		hash ^= (hash >> 6);
	}
	hash += (hash << 3);
	hash ^= (hash >> 11);
	hash += (hash << 15);
	return (hash & mask);
}

static unsigned int gethash(Lptr Snode)
{
	int index;

	switch (Vtype(Snode)) {
	case STRTYPE:
	{
		char *strP = Strdata((Start_frag(Snode)));
		index = hash(strP, strlen(strP));
		break;
	}
	case INTTYPE:
		index = hash( (char*) &(Intval(Snode)), sizeof(int));
		break;
	default:
		fprintf(stderr, "Attempting to gethash for %s\n", Vname(Snode));
		index = 0;
		assert(0);
	}
	return(index);
}

/* Initialize the hash table and set all entries to be NULL */
void initializeHash(void) {
	int h;

	for (h = 0; h < MAXHASHSIZE; ++h) {
		// The hash table is nil node terminated so that we can detect atoms not in the hash table efficiently
		// These will be the atoms having a hashnext pointer of 0.
		hashtable[h] = getNil();
	}
	initialized = 1;
}

Lptr firstatom(void)
{
	if (initialized) {
		Lptr	ret;
		int 	h;

		for (h = 0; h < MAXHASHSIZE; ++h) {
			ret = hashtable[h];
			if (Vtype(ret) != NILTYPE) {
				return(ret);
	}	}	}
	return(getNil());
}

Lptr nextatom(Lptr Snode)
{
	assert(Vtype(Snode) == STRTYPE || Vtype(Snode) == INTTYPE);

	if (initialized) {
		Lptr	ret;
		int 	h;

		ret = Nexthash(Snode);
		if (Vtype(ret) != NILTYPE) {
			return(ret);
		}
		for (h = gethash(Snode); ++h < MAXHASHSIZE; ) {
			ret = hashtable[h];
			if (Vtype(ret) != NILTYPE) {
				return(ret);
	}	}	}
	return(getNil());
}

void finalizeHash(void) {
	int 	h;
	Lptr	P, nextP;

	if (initialized) {
		for (h = 0; h < MAXHASHSIZE; ++h) {
			for (P = hashtable[h]; Vtype(P) != NILTYPE; P = nextP) {
				nextP = Nexthash(P);
				release(P);
			}
			hashtable[h] = getNil();
}	}	}

/* Add an atom node to the hash table
   =========================================
   Parameter: Snode is the atom node
   =========================================
   Required:  Snode must be an int or string
*/	 
void addhash(Lptr Snode) {
	unsigned int index;
	
	/* if the hash table not initialized, first initialize it */
	if (!initialized) {
		initializeHash();
	}
	index = gethash(Snode);
	Nexthash(Snode) = hashtable[index];
	hashtable[index] = assign(Snode);	 
}

/* Delete a node from the hash table
   ============================================================
   Parameter:
	 Snode	 the node to be deleted
   =============================================================
   Required:
	 Snode is either int or string
	 Snode is already in the hash table (already added into it)
*/		 

void remob(Lptr Snode){
	unsigned int index;
	Lptr	sptr, *sptrP;

	if (initialized && Nexthash(Snode)) {
		index = gethash(Snode); 
		for (sptrP = hashtable + index; sptr = *sptrP; sptrP = &(Nexthash(sptr))) {
			if (sptr == Snode) {
				*sptrP = Nexthash(Snode);
				Nexthash(Snode) = 0;
				release(Snode);
				return;
	}	}	}
	fprintf(stderr, "Error: remob on a non-existant %s.\n", Vname(Snode));
}


/* Given a string, find the hash table entry
   ==========================================
   Parameter:
	 stringval is the string value 
*/
Lptr findhashstr(char *stringval) {
	Lptr tempf;
	unsigned int index;
	unsigned int lth;

	if (!initialized) {
		return NULL;
	}

	lth = (unsigned int) strlen(stringval);
	if (lth > STRSIZE) {
		lth = STRSIZE;
	}

	/* calculate the hash value and the index */
	index = hash(stringval, lth);

	/* go through the nexthash chain */
	for (tempf = hashtable[index]; ; tempf = Nexthash(tempf) ) {
		switch (Vtype(tempf)) {
		case STRTYPE:
			if (stringcmp(Start_frag(tempf), stringval) == 0) {
				return tempf; 
			}
			break;
		case NILTYPE:
			return NULL;
	}	}
}

// --------------------------------
// Start of what used to be nlisp.c
// --------------------------------

/****************************************************************************
   Function for the Nodes
 ****************************************************************************/

//Ignore the if(vAtom(car.... line, it should be removed
void setdef(Lptr Sc, Lptr Sv) {
	if (vAtom(Sc)){
		setSptr(&(Def(Sc)), Sv);
	}
}

//Ignore the if(vAtom(car.... line, it should be removed
Lptr getdef(Lptr Sc) {

	if (vAtom(Sc)) {
		return(Def(Sc));
	}	
	return (getNil());
}

static int frag_sizeof(void)
{
	return(sizeof(fragT));
}

static celltypeE frag_stype(void)
{
	return(FRAGTYPE);
}

static const char * frag_name(void)
{
	return("FRAG");
}

static void frag_dump(void *P, FILE *F)
{
	Fptr tmpSource;

	fprintf(F, "%p Frag", P);
	for (tmpSource = (Fptr) P; tmpSource != NULL ; tmpSource = Next_frag(tmpSource)) {
		fprintf(F, " \"%s\"", Strdata(tmpSource));
	}
}
static int frag_empty(void *frag)
{
	return(1);
}

static void frag_set_mark(Lptr fnode);
static void frag_clr_mark(Lptr fnode);

static vtableT frag_vtable =
{
	&frag_sizeof,
	FRAGTYPE,
	&frag_name,
	&frag_dump,
	&frag_empty,
	&frag_set_mark,
	&frag_clr_mark,
	&is_false
};

static void frag_set_mark(Lptr fnode)
{
	fprintf(stderr, "Can't mark fragments\n");
}

static void frag_clr_mark(Lptr fnode)
{
}

//Takes a string and put it in the Str store cells in chunks of STRSIZE characters
Lptr stringToStrStore(char *Stringval) 
{
	size_t length,i;
	Lptr initialptr, nptr, temp;

	length = strlen(Stringval);
	initialptr = (Lptr) getMemBlock(&frag_vtable);

	strncpy(Strdata(initialptr), Stringval,STRSIZE);
	Strdata(initialptr)[STRSIZE] = '\0';
	
	nptr=initialptr;
	for (i=STRSIZE ; i<length ; i +=STRSIZE, nptr=Next_frag(nptr)){
		temp = (Lptr) getMemBlock(&frag_vtable);
		strncpy(Strdata(temp),&(Stringval[i]),STRSIZE);
		Strdata(temp)[STRSIZE] = '\0';
		Next_frag(nptr) = temp;
	}
	Next_frag(nptr) =  NULL;

	return initialptr;
}

unsigned int stringLth(Lptr atomstr)
{
	Lptr			frag, next_frag;
	unsigned int	lth;

	lth = 0;

	for (frag = atomstr; frag != NULL ; frag = next_frag) {
		next_frag = Next_frag(frag);
		if (next_frag) {
			lth += STRSIZE;
		} else {
			lth += (unsigned int) strlen(Strdata(frag));
	}	}
	return(lth);
}

/*Receives the StringStore from an vAtom and copy it to a String in Stringval. The StringVal array must exist before calling
  this function.
  */
void strStoreToString(Lptr atomstr, char* Stringval) {
	Lptr		frag;
	char		*dest;
	const char	*source;
	const char	*max;

	
	dest = Stringval;
	for(frag = atomstr; frag != NULL ; frag = Next_frag(frag)) {
		source = Strdata(frag);
		max    = source + sizeof(Strdata(frag));
		for (; source < max && (*dest = *source++); ++dest);
	}
	*dest = '\0';
}

//Compare a StringStore to a String (in an array)
int stringcmp(Lptr Strblk, char *strval) {
	int i,j, maxsize = STRSIZE-1, morestr, compare;
	for (i = 0, j = 0, morestr = (strval[0] != '\0'), compare = 0; (compare == 0) && morestr; ) {
		compare = Strdata(Strblk)[i] - strval[j];
		if (!compare) {
			if ( (i == maxsize) && (Next_frag(Strblk) != NULL) ) {
				i=0;
				Strblk = Next_frag(Strblk);
				morestr = ( (strval[j++] != '\0') && (Strdata(Strblk)[i] != '\0') );
			}
			else
				morestr = ( (strval[j++] != '\0') && (Strdata(Strblk)[i++] != '\0') );
		}
	}
	return compare;
}

//Compare two StringStores
int storecmp(Lptr Strblk1, Lptr Strblk2) {
	int i,j, morestr, compare;
	for (i = 0, j = 0, morestr = 1, compare = 0; (compare == 0) && morestr; ) {
		compare = Strdata(Strblk1)[i] - Strdata(Strblk2)[j];
		if (!compare) {
			if (i == STRSIZE) {
				if (Next_frag(Strblk1) != NULL)  {
				i=0;
				Strblk1 = Next_frag(Strblk1);
				}
				if ( (j == STRSIZE) && (Next_frag(Strblk2) != NULL) ) {
				j=0;
				Strblk2 = Next_frag(Strblk2);
				}
			}
			else
				morestr = ( (Strdata(Strblk1)[i++] != '\0') && (Strdata(Strblk2)[j++] != '\0') );
		}
	}
	return compare;
}

//Just for debugging. It displays stats about the free memory
void finish(void)
{
	printAllFreeStats();
}

/* Get the integer value of an atom node of
   integer type
*/
int getAtomIntVal(Lptr Snode, int *valP) {
	if (Vtype(Snode) == INTTYPE) {
		*valP = Intval(Snode);
		return(1);
	}
	fprintf(stderr, "getAtomIntVal can't return an integer when presented with a %s cell\n", Vname(Snode));
	return(0);
}

/* Get the pointer to the string storage of
   an atom node of string type
*/
int getAtomStrStore(Lptr Snode, Lptr *resultPP) {
	if (Vtype(Snode) == STRTYPE) {
		*resultPP = Start_frag(Snode);
		return(1);
	}
	fprintf(stderr, "getAtomStrStore can't return a string when presented with a %s cell\n", Vname(Snode));
	*resultPP = getNil();
	return(0);
}

/* Recursively returns vcar and vcdr acording to the string passed.
   For example, cadddar = car(cdr(cdr(cdr(car()))))
   ================================================================
   Parameters:
	 v is the c[a|d]*r string;
	 Snode is the node to apply v
   ================================================================
   Returns the node that is the result of applying v to Snode
   ================================================================
   Note: cvr is called recursively
*/	 

Lptr icvr(char* v, Lptr Snode) {
	size_t vlen;
	vlen = strlen(v);

	if (v[1] == 'r') {
		if (v[0] == 'a') {
			return ( icar(Snode) );
		}
		//v[0]== 'd'
		return ( icdr(Snode) );
	}
	if (v[0] == 'a') {
		return ( icar(icvr(++v,Snode)) );
	}
	//v[0]== 'd'
	return ( icdr(icvr(++v,Snode)) );
 }

Lptr cvr(char* v, Lptr Snode) {
	size_t vlen;
	vlen = strlen(v);
   
	if (v[1] == 'r') {
		if (v[0] == 'a') {
			return ( car(Snode) );
		}
		//v[0]== 'd'
		return ( cdr(Snode) );
	}
	if (v[0] == 'a') {
		return ( car(cvr(++v,Snode)) );
	}
	//v[0]== 'd'
	return ( cdr(cvr(++v,Snode)) );
 }

/* Checks wether the node is atom (int or string)
   ====================================================
   Parameter:
	 Snode	   the node to check
   =====================================================
   Returns true if Snode is an int or string, false o.w.
*/	 
boolean vAtom(Lptr Snode){
	switch (Vtype(Snode)) {
	case INTTYPE:
	case STRTYPE:
		return TRUE;
	}
	return FALSE;
}

/* Checks wether the node is the nil cell
   ====================================================
   Parameter:
	 Snode	   the node to check
   =====================================================
   Returns true if Snode is nil, false o.w.
*/	
boolean vNull(Lptr Snode){
	switch (Vtype(Snode)) {
	case NILTYPE:
		return TRUE;
	}
	return FALSE;
}

/* Checks wether the node is a cons cell or nil cell
   ====================================================
   Parameter:
	 Snode	   the node to check
   =====================================================
   Returns true if Snode is a cons cell or nil, false o.w.
*/	
boolean vListp(Lptr Snode){

	switch (Vtype(Snode)) {
	case CONSTYPE:
	case NILTYPE:
		return TRUE;
	}
	return FALSE;
}

/* Assign the def pointer of Snode to be S1node */
Lptr set(Lptr Snode, Lptr Slnode){
	switch (Vtype(Snode)){
	case INTTYPE :
	case STRTYPE :
		setdef(Snode,Slnode);
		break;
	default:
		assert(0);
		assign(Slnode);
		release(Slnode);
	}
	return (Snode);
}

/* Given a String (no spaces, only 1 token) makes the atom with that string.
   =========================================================================
   Parameter:
	 Stringval	 the input string
   =========================================================================
   Returns the atom node that is built.
   =========================================================================
   Note:
	 if it exist in the hash, return it, otherwise, create a new one and
	 add it to the hash.
*/

static int int_sizeof(void)
{
	return(sizeof(intT));
}

static celltypeE int_stype(void)
{
	return(INTTYPE);
}

static const char * int_name(void)
{
	return("INT");
}

static void int_dump(void *P, FILE *F)
{
	fprintf(F, "%p Integer %i refcnt=%d",  P, Intval((Mptr) P), Rcount((Mptr) P));
}

static int	int_empty(void *Snode)
{
	release(Def((Mptr) Snode));
	return(1);
}

static void int_set_mark(Lptr fnode);
static void int_clr_mark(Lptr fnode);

static vtableT int_vtable =
{
	&int_sizeof,
	INTTYPE,
	&int_name,
	&int_dump,
	&int_empty,
	&int_set_mark,
	&int_clr_mark,
	&is_false
};

static vtableT int_vtable_marked =
{
	&int_sizeof,
	INTTYPE,
	&int_name,
	&int_dump,
	&int_empty,
	&int_set_mark,
	&int_clr_mark,
	&is_true
};

static void int_set_mark(Lptr fnode)
{
	Vtable(fnode) = &int_vtable_marked;
}

static void int_clr_mark(Lptr fnode)
{
	Vtable(fnode) = &int_vtable;
}

Lptr
mkintatom(int val)
{
	Lptr	temp;

	temp            = (Lptr) getMemBlock(&int_vtable);
	Rcount(temp)	= 0;
	Nexthash(temp)	= NULL;
	Def(temp)		= getUndef();
	Intval(temp)    = val;
	return temp;
}

static int str_sizeof(void)
{
	return(sizeof(stringT));
}

static celltypeE str_stype(void)
{
	return(STRTYPE);
}

static const char * str_name(void)
{
	return("STRING");
}

static void str_dump(void *P, FILE *F)
{
	Fptr tmpSource;

	fprintf(F, "%p String", P);
	for (tmpSource = Start_frag((Mptr) P); tmpSource != NULL ; tmpSource = Next_frag(tmpSource)) {
		fprintf(F, " \"%s\"", Strdata(tmpSource));
	}
	fprintf(F, " refcnt=%d", Rcount((Mptr) P));
}

static int str_empty(void *Snode)
{
	Fptr	frag, next;

	release(Def((Mptr) Snode));

	for (frag = Start_frag((Mptr) Snode); frag != NULL; frag = next) {
		next = Next_frag(frag);
		// Call this even if it is a no-op to ensure we are not freeing free memory or anything
		if (Vempty(frag)) {
			freeMemBlock((freeT *) frag);
	}	}
	return(1);
}

static void str_set_mark(Lptr fnode);
static void str_clr_mark(Lptr fnode);

static vtableT str_vtable =
{
	&str_sizeof,
	STRTYPE,
	&str_name,
	&str_dump,
	&str_empty,
	&str_set_mark,
	&str_clr_mark,
	&is_false
};

static vtableT str_vtable_marked =
{
	&str_sizeof,
	STRTYPE,
	&str_name,
	&str_dump,
	&str_empty,
	&str_set_mark,
	&str_clr_mark,
	&is_true
};

static void str_set_mark(Lptr fnode)
{
	Vtable(fnode) = &str_vtable_marked;
}

static void str_clr_mark(Lptr fnode)
{
	Vtable(fnode) = &str_vtable;
}

Lptr
mkstringatom(char *Stringval)
{
	Lptr temp;

	if ((temp = findhashstr(Stringval)) == NULL) {
		temp             = (Lptr) getMemBlock(&str_vtable);
		Rcount(temp)	 = 0;
		Def(temp)		 = getUndef();
		Nexthash(temp)   = NULL;
		Start_frag(temp) = stringToStrStore(Stringval);
		addhash(temp);
	}
	return temp;
}

Lptr 
mkatom(char *Stringval){
	int val;
	Lptr temp;

	val = atoi(Stringval);	 /* see whether it is a int */
	
	if ((val == 0) && (Stringval[0]!='0')) { //is string
		temp = mkstringatom(Stringval);
	}
	else { //is numeric
		temp = mkintatom(val);
	}
	return temp;
}

Lptr
getTrue(void)
{
	return(mkstringatom("t"));
}

/* Try to create the list from the expression begining with (
   ===========================================================
   Parameters:
	 Stringval	the expression in string format
	 pos		the position in the string.
				The parameter is needed because of the recursive
				calling
   ============================================================
   Returns		the cons cell  
*/
Lptr getList(char *Stringval, int*pos) {
	tokentypeE ttype; 
	int  strlength= (int) strlen(Stringval);
	Lptr tempS, Initial, lastS, newS;

	/* get next token */
	tempS = getToken(&ttype, Stringval, pos); 
	switch (ttype) {
	case CLOSEPRTS: 	/* "()": a nil cell */
		return (getNil());
	case OPENPRTS:		/* "((..." call recursively */
		tempS = getList(Stringval,pos);
		break;
	case NUMBER:
	case LETTER:
	case SPECIALCHR:
		break;
	default:
		tempS = NULL;
	}

	/* tempS is expected to be the node built for the subexpression */	  
	if (tempS == NULL) {	  /* "(" without close matching: */
		return NULL;
	}
   
	Initial = cons(tempS, getNil());
	lastS=Initial;

	/* iterate over the expression */

	for (;;) {
		/* get the next token */
		tempS = getToken(&ttype, Stringval, pos);

		switch (ttype) {
		case OPENPRTS :   /* if it is "(((...", recurse again */
			tempS = getList(Stringval, pos);
			if (tempS == NULL) {
				goto fail;
			}
		case NUMBER :
		case LETTER :
		case SPECIALCHR:
			/* add cons(tempS, nil) to the original nil cdr of lastS */
			newS  = cons(tempS, getNil());
			lastS = irplacd(lastS,newS);
			lastS = newS;
			break;

		case CLOSEPRTS :  /* find a ')' match of '(': */
			return Initial;
		default:
			goto fail;
		}
	}

fail:
	/* cannot find the correct match of ')', cleanup and return */
	release(Initial);
	return NULL;
}

/* lread is the main function to create Sexpressions from a string.
   It calls mkatom and getlist depending on the begining
   of the string. If the recursive functions fail, this one will
   also fail. At the end it will try to print where the error occurred
   if found.
   ====================================================================
   Note: calls getList as a helper function
*/	 
Lptr lread(char *Stringval) {
	tokentypeE tokentype;
	int chrp=0,i;
	Lptr head;

	head = getToken(&tokentype,Stringval, &chrp);
	switch (tokentype) {
	case NUMBER:
	case LETTER:
	case SPECIALCHR:
		break;
	case OPENPRTS:
		head = getList(Stringval, &chrp);
		break;
	default:
		head = NULL;
	}

	if (head != NULL) {
		getToken(&tokentype,Stringval, &chrp);
		if (tokentype == WHITESPC) {
			return head;
	}	}
	fprintf(stderr,"Error reading string below at position %i\n%s\n",chrp,Stringval);
	for (i=1;i<chrp;i++) fprintf(stderr," "); 
	fprintf(stderr,"^\n");
	if (head) {
		assign(head);
		release(head);
	}
	return NULL;
}

/*******************************
		 NEW OPERATIONS
********************************/
/* compare the string with the string store of a node
   ====================================================
   Parameters:
	 S2 	   the node
	 stringtc  the string to compare
   ====================================================
   Return false if the strings do not match or S2 is not
   a string cell, true o.w.
*/	   
boolean eql(Lptr S2, char* stringtc) {
	if (Vtype(S2) == STRTYPE)
		return (stringcmp(Start_frag(S2),stringtc) == 0);
	else
		return (FALSE);
}


/* Compare wether the two nodes have equal values
   ===============================================
   Parameters: E1, E2 two nodes to compare
   ===============================================
   Returns true if they have equal values, false
   o.w.
   ===============================================
   Note: if two nodes are at the same address, they
		 are equal;
		 if they are of different types, they are
		 not equal		   
*/	 
boolean eq(Lptr E1, Lptr E2) {
	
	if (E1 == E2) {
		return TRUE;
	}
	if ((Vtype(E1)) == (Vtype(E2))) {
		switch (Vtype(E1)) {
		case INTTYPE :
			return ((Intval(E1)) == (Intval(E2)));

		case STRTYPE :
			return (E1 == E2);
	}	}
	return FALSE;
}

/* the negation of eq above */
boolean neq(Lptr E1, Lptr E2) {
	return (!eq(E1, E2));
}

/* Compare two atom nodes. If one is integer and one string, casting is
   done for the comparison.
   ====================================================================
   Returns 0 if equal, o.w. the difference of them
   ====================================================================
   Note: for integers, the difference is the diff of their values;
		 for string, it is the diff of the first pair of diff chars.
*/	 
int atomcmp(Lptr atom1, Lptr atom2) {
	if (vAtom(atom1) && vAtom(atom2)) {
		if (Vtype(atom1) == INTTYPE)
			if (Vtype(atom2) == INTTYPE)
				return ((Intval(atom1)) - (Intval(atom2)));
			else {	  /* atom2 is a string, casting is needed */
				char intvalue[20];
				sprintf(intvalue,"%d",(Intval(atom1)));
				return (-stringcmp(Start_frag(atom2),intvalue));
			}
		else  /* atom1 is a string */
			if (Vtype(atom2) == INTTYPE) {	/* casting int to string */
				char intvalue[20];
				sprintf(intvalue,"%d",(Intval(atom2)));
				return (stringcmp(Start_frag(atom1),intvalue));
			} 
			else
				return (storecmp(Start_frag(atom1), Start_frag(atom2)));
	}
	return(-1000);
}

/* Compare two nodes Snode1 and Snode2.
   Returns 0 if they are equal, o.w. the difference
*/
int compareS(Lptr Snode1, Lptr Snode2) {
	int carcmp;    
	switch (Vtype(Snode1)) {
		case INTTYPE :
		case STRTYPE :
		switch (Vtype(Snode2)) {
			case INTTYPE :
			case STRTYPE :
				return (atomcmp(Snode1, Snode2));
			break;

			case CONSTYPE :
				return (- 2000);// Snode1 < Snode2 :: atom < cons
			break;
				
			case UNDEFTYPE:
			case ERRORTYPE:
			case NILTYPE :
				return (2000);// Snode1 > Snode2 :: atom > nil
			break;
		}
		break;
		case CONSTYPE :
		switch (Vtype(Snode2)) {
			case INTTYPE :
			case STRTYPE :
				return (2000);//Snode1 > Snode2 :: cons > atom
			break;
			case CONSTYPE :
				carcmp = compareS(car(Snode1), car(Snode2));
				if (carcmp != 0)
					return (carcmp);
				else
					return (compareS(cdr(Snode1), cdr(Snode2)));
			break;
			case UNDEFTYPE:
			case ERRORTYPE:
			case NILTYPE :
				return (2000);// Snode1 > Snode2 :: cons > nil
			break;
		}
		break;

		case NILTYPE :
		case UNDEFTYPE:
		case ERRORTYPE:
			if (vNull(Snode2))
				return (0); 		   
			else
				return (-2000); // Snode1 < Snode2 :: nil < atom|cons
		break;	
	}
	assert(0);
	return(0);
}

/* Given a prefix, make a symbol and generate an atom
   ===================================================
   Parameter: newprefix the prefix string
   ===================================================
   Returns the generated atom node
*/

static unsigned int gensym_count=1;

void gensym_init(void)
{
	gensym_count = 1;
}

Lptr gensym(char* newprefix) {
	static char prefix[16] = "G";
	char temp[32]= "";	  
	
	if (strlen(newprefix) > 0)
		strcpy(prefix,newprefix);
	sprintf(temp, "%s%u", prefix, gensym_count++);
	return (mkstringatom(temp));
}

/* Get the property value
   ========================================
   Parameters:
	symbol		  the property list
	indicator	  the property name(s)
   ========================================
   Returns the property value 
*/

Lptr assq(Lptr list, Lptr indicator)
{
	Sptr	ind = indicator;

	switch (Vtype(ind)) {
		case CONSTYPE:
		{
			Lptr	rest, ret;

			ind  = car(indicator);
			if (Vtype(ind) == CONSTYPE) {
				// We expect the indicators to be a simple list
				break;
			}
			rest = cdr(indicator);
			if (Vtype(rest) != NILTYPE) {
				ret = assq(list, ind);
				ind = cdr(indicator);
				if (Vtype(ind) == NILTYPE) {
					return(ret);
				}
				return(assq(ret, rest));
			}
		}
		case STRTYPE:
		case INTTYPE:
		{
			Lptr	at;

			for (at = list; Vtype(at) == CONSTYPE; at = cdr(cdr(at))) {
				if (eq(car(at), ind)) {
					/* name matching, car(property) has the property name
						the property name/value are encoded by two cons cells,
						car(cdr(property)) has the property value 
					*/
					return (car(cdr(at)));
				}
				/* go to the next property, cdr(cdr(property) skips
				the value of the current property and returns the
				cons cell of the next property name 
				*/
			}
			break;
		}
	}
	return (getNil());
}

/* Get the property value
   ========================================
   Parameters:
	symbol		  the property list owner
	indicator	  the property name
   ========================================
   Returns the property value 
*/
Lptr getprop(Lptr symbol, Lptr indicators) 
{
	if (Vtype(symbol) == STRTYPE) {
		return (assq(Def(symbol), indicators));
	}
	return(getNil());
}

/* Put a property name/value pair into the property list
   ======================================================
   Parameters:
	symbol			the property list owner
	indicator		the property name
	value			the property value
   ======================================================
   Returns the pointer to the head of the new property list
*/	 

Lptr putp(Lptr header, Lptr indicator, Lptr value, int commit)
{
	Sptr	ind;

	ind = indicator;

	switch (Vtype(ind)) {
		case CONSTYPE:
		{
			Lptr	rest, at, ret, header1, header2;

			ind = car(indicator);
			if (Vtype(ind) == CONSTYPE) {
				// Indicators should form a simple list
				break;
			}
			rest = cdr(indicator);
			if (Vtype(rest) != NILTYPE) {
				
				for (at = header; Vtype(at) == CONSTYPE; at = cdr(cdr(at))) {
					if (eq(car(at), ind)) {
						at      = cdr(at);
						header1 = car(at);
						ret = putp(header1, rest, value, commit);
						if (ret != header1) {
							if (commit) {
								prplaca(at, ret);
							} else {
								rplaca(at, ret);
						}	}
						return(header);
				}	}
				// This indicator does not exist in the property list
				header2 = putp(getNil(), rest, value, commit); 
				if (Vtype(header2) != NILTYPE) {
					if (Vtype(header) == UNDEFTYPE) {
						// Change the previously undefined value to the empty list
						header = getNil();
					}
					header1 = cons(ind, cons(getNil(), header));
					at = cdr(header1);
					if (commit) {
						prplaca(at, header2);
					} else {
						rplaca(at, header2);
					}
					return(header1);
				}
				return(header);
		}	}
	case STRTYPE:
		case INTTYPE:
		{
			Lptr	at;

			for (at = header; Vtype(at) == CONSTYPE; at = cdr(cdr(at))) {
				if (eq(car(at), ind)) {
					/* name matching, car(property) has the property name
						the property name/value are encoded by two cons cells,
						car(cdr(property)) has the property value 
					*/
					at = cdr(at);
					if (commit > 0) {
						prplaca(at, value);
					} else {
						rplaca(at,value);
					}
					return(header);
				}
				/* go to the next property, cdr(cdr(property) skips
				the value of the current property and returns the
				cons cell of the next property name 
				*/
			}
			if (Vtype(value) != NILTYPE) {
				// Create a new property pair
				if (Vtype(header) == UNDEFTYPE) {
					header = getNil();
				}
				header = cons(ind, cons(getNil(), header));
				// Do it this way so that versioning can unwind
				at = cdr(header);
				if (commit) {
					prplaca(at, value);
				} else {
					rplaca(at, value);
			}	}
			return(header);
	}	}
	return (getNil());
}

Lptr pputprop(Lptr symbol, Lptr indicator, Lptr value) 
{
	Lptr def, header;

	def    = Def(symbol);
	header = putp(def, indicator, value, 1);
	if (header != def) {
		setSptr(&Def(symbol), header);
	}
	return(header);
}

Lptr putprop(Lptr symbol, Lptr indicator, Lptr value) 
{
	Lptr	def, header;

	def    = Def(symbol);
	header = putp(def, indicator, value, 0);
	if (header != def) {
		setSptr(&Def(symbol), header);
	}
	return(header);
}

/* Remove a property with a specified name from the property list
   ===============================================================
   Parameters:
	 symbol 	  the property list owner
	 indicator	  the property name
   ===============================================================
   Returns the property list if the property with
   matching name is found and removed, o.w. returns a nil
*/	 

Lptr remprop(Lptr symbol, Lptr indicator) 
{
	Lptr ret;

	ret = getprop(symbol, indicator);
	if (Vtype(ret) == NILTYPE) {
		return(ret);
	}
	ret = putprop(symbol, indicator, getNil());
	return(ret);
}

Lptr append(Lptr x, Lptr y) { //Both arguments need to be lists or it will not work
	if (vNull(x))
		return(y);
	else
		return (cons(car(x),append(cdr(x),y)));
}

Lptr last(Lptr inList) {
	Lptr lastcons;
	
	if (!vListp(inList))
		return (getNil());
	else
		for (lastcons = inList; !vNull(cdr(lastcons)); lastcons=cdr(lastcons));
	return (lastcons);	  
}

Lptr insertSortList(Lptr Head, Lptr Element) {
	Lptr temp, cdrTemp;
	int cmpS;

	temp = Head;
	cdrTemp = cdr(temp);
	cmpS = compareS(car(cdrTemp),Element);
	while(!vNull(cdrTemp) && cmpS < 0) {
		temp = cdrTemp;
		cdrTemp = cdr(temp);
		cmpS = compareS(car(cdrTemp),Element);
	}

	if (vNull(cdrTemp) || cmpS > 0) // by ignoring the case cmpS = 0 we are removing duplicates
		rplacd(temp, cons(Element, cdr(temp)));
	return (Head);
}

Lptr listUnion(Lptr L1, Lptr L2) {
	Lptr UListHead, temp;

	UListHead = assign(cons(getNil(), getNil()));
	for (temp = L1; !vNull(temp); temp= cdr(temp))
		insertSortList(UListHead, car(temp));
	for (temp = L2; !vNull(temp); temp= cdr(temp))
		insertSortList(UListHead, car(temp));
	temp = UListHead;
	UListHead = assign(cdr(UListHead));
	release(temp);
	return(decrement(UListHead));
}

boolean member(Lptr x, Lptr y) {
	return ( !vNull(y) && (eq(x,car(y)) || member(x,cdr(y))) );
}


/************************************************************
				GENERIC GET/PUT/REM OPERATIONS
*************************************************************/

Lptr findPrevList(Lptr List, Lptr Node) {
	Lptr temp;
	for (temp=List; !vNull(temp) && !eq(car(cdr(temp)), Node); temp=cdr(temp));
	return (temp);
}

Lptr findListHead(Lptr List, Lptr Node) {
	Lptr temp;
	for (temp=List; !vNull(temp) && !eq(car(temp), Node); temp=cdr(temp));
	return (temp);
}

Lptr addList (Lptr ListHead, Lptr Node) {
	return(rplaca(ListHead,cons(Node,car(ListHead))));
}

Lptr removeList (Lptr ListHead, Lptr Node) {
	Lptr PrevHead;
	if (eq(Node, car(car(ListHead))))
		return(rplaca(ListHead,cdr(car(ListHead))));
	//else
	PrevHead = findPrevList(car(ListHead),Node);
	return( rplacd(PrevHead, cdr(cdr(PrevHead))) );
}

Lptr updateList(Lptr ListHead, Lptr oldNode, Lptr newNode) {
	return(rplaca(findListHead(car(ListHead),oldNode),newNode));
}

Lptr getprophead(Lptr symbol, Lptr indicator) {
	return( getph_fd(symbol, indicator, &eval) );	 
}

Lptr getph_fd(Lptr symbol, Lptr indicator, Lptr (*feval) (Lptr)) {
	Lptr property;

	property = (*feval)(symbol);
	while (!vNull(property))
		if (eq(car(property),indicator))
			return (cdr(property));
		else
			property = cdr(cdr(property));
	return (getNil());
}


/******************************************************************************
  Parser functions
 ******************************************************************************/



tokentypeE getCharType(char c) {

	if ( (c>='a' && c <= 'z') || (c >= 'A' && c <= 'Z') )
		return LETTER;
	if (c >= '0' && c <= '9')
		return NUMBER;
	switch (c){
		case '(' :
		   return OPENPRTS;
		case ')' :
		   return CLOSEPRTS;
		case ' ' :
		case '\t' :
		case '\r' :
		case '\n' :
		   return WHITESPC;
		case '!' :
		case '*' :
		case '+' :
		case '-' :
		case '\'' :
		case '>' :
		case '<' :
		case '?' :
		   return SPECIALCHR;
   }
   return ERROR;
}

Lptr 
getToken(tokentypeE *toktype, char *inputstr, int *pos) 
{
	tokentypeE	type;
	int 		c, c1, quotes;
	int 		val;
	int 		first, last;
	char		*bufferP, *P;
	Lptr		retP;
	int 		plus;

	retP	 = 0;
	plus	 = 0;
	quotes	 = -1;
	for (;;) {
		c	 = inputstr[(*pos)++];
		switch (c) {
		case 0:
			type = WHITESPC;
			--(*pos);
			goto done;
		case '"':
			type   = LETTER;
			quotes = 1;
			c	   = inputstr[(*pos)++];
			break;
		case ';':
			// Comment to end of line
			while (c = inputstr[(*pos)++]) {
				if (c == '\n') {
					break;
			}	}
			continue;
		case '-':
			c1 = inputstr[*pos];
			if (!plus && c1 >= '0' && c1 <= '9') {
				type = NUMBER;
				++(*pos);
				plus = -1;
			} else {
				type = SPECIALCHR;
			}
			break;
		default:
			type = getCharType(c);
		}
		if (!plus) {
			plus = 1;
		}
		switch (type) {
		case WHITESPC:
			continue;
		case OPENPRTS:
		case CLOSEPRTS:
			goto done;
		case NUMBER:
			for (val = 0;;)  {
				val = val * 10 + c - '0';
				c = inputstr[(*pos)];
					if (c < '0' || c > '9') {
					break;
				}
				++(*pos);
			}
			if (plus < 0) {
				val = -val;
			}
			retP = mkintatom(val);
			goto done;
		case LETTER:
			last  = *pos;
			first = last-1;
			for (;; ++last) {
				c = inputstr[last];
				switch (c) {
				case '"':
					if (quotes > 0) {
						++quotes;
						continue;
					}
					goto have_string;
				case '\'':
					if (quotes > 0) {
						if (--quotes) {
							continue;
					}	}
					goto have_string;
				case '\\':
					if (inputstr[++last]) {
						continue;
					}
					--last;
					goto have_string;
				}
				
				if (quotes > 0) {
					continue;
				}
				switch (getCharType(c)) {
				case NUMBER:
				case LETTER:
				case SPECIALCHR:
					continue;
				}
				break;
			}
			break;
		case SPECIALCHR:
			last  = *pos;
			first = last-1;
			for (;;++last) {
				c = inputstr[last];
				switch (getCharType(c)) {
				case NUMBER:
				case SPECIALCHR:
					continue;
				}
				break;
			}
			break;
		default:
			goto done;
		}
		break;
	}

have_string:

	// If arrive here have a string starting at first and ending at the character before last

	bufferP = P = _alloca(last-first+1);
	if (!bufferP) {
		fprintf(stderr, "Can't alloca(%d)\n", last-first);
		goto done;
	}
	while (first < last) {
		c = inputstr[first++];
		if (c == '\\') {
			c = inputstr[first++];
			switch (c) {
			case 'r':
				c = '\r';
				break;
			case 'n':
				c = '\n';
				break;
			case 't':
				c = '\t';
				break;
			case 'f':
				c = '\f';
				break;
		}	}
		*P++ = c;
	}
	*P	 = 0;
	retP = mkstringatom(bufferP);
	if (quotes != -1) {
		// Skip the trailing quote
		++last;
	}
	*pos = last;

done:
	*toktype = type;
	return(retP);
}

/************************************************************************
  Evaluation Functions
 ************************************************************************/

/***************************************************************************
Description: add atom1 and atom2  
Input parameters: atom1 and atom2 are operands 
Returns: an atom that contains the sum 
Note: assuming the operands are INTTYPE 
***************************************************************************/

evalOkE
args0(Lptr Sexpr)
{
	if (!vNull(Sexpr)) {
		return(eval_too_many);
	}
	return(eval_ok);
}

evalOkE
args1(Lptr Sexpr, Lptr *arg1P)
{
	if (vNull(Sexpr)) {
		return(eval_too_few);
	}
	*arg1P = car(Sexpr);
	if (!vNull(cdr(Sexpr))) {
		return(eval_too_many);
	}
	return(eval_ok);
}

evalOkE
args1_2(Lptr Sexpr, Lptr *arg1P, Lptr *arg2P)
{
	Lptr	args;
	
	if (vNull(Sexpr)) {
		return(eval_too_few);
	}
	*arg1P = car(Sexpr);
	args   = cdr(Sexpr);
	if (vNull(args)) {
		*arg2P = 0;
	} else {
		*arg2P = car(args);
		if (!vNull(cdr(args))) {
			return(eval_too_many);
	}	}
	return(eval_ok);
}

evalOkE
args2(Lptr Sexpr, Lptr *arg1P, Lptr *arg2P)
{
	if (vNull(Sexpr)) {
		return(eval_too_few);
	}
	*arg1P = car(Sexpr);
	return(args1(cdr(Sexpr), arg2P));
}

evalOkE
args3(Lptr Sexpr, Lptr *arg1P, Lptr *arg2P, Lptr *arg3P)
{
	if (vNull(Sexpr)) {
		return(eval_too_few);
	}
	*arg1P = car(Sexpr);
	return(args2(cdr(Sexpr), arg2P, arg3P));
}

static evalOkE
eval_abortver(Lptr Sexpr, Lptr *resultPP)
{
	evalOkE	get;

	get = args0(Sexpr);
	if (get == eval_ok) {
		if (!getCurrentVersion()) {
			return(eval_cant_abort);
		}
		abortVer();
		*resultPP = mkintatom(getCurrentVersion());
	}
	return(get);
}

static evalOkE
eval_add(Lptr Sexpr, Lptr *resultPP) {
	Lptr	arg1, arg2;
	int 	val1, val2;
	int 	ret;
	evalOkE	get;

	get = args2(Sexpr, &arg1, &arg2);
	if (get == eval_ok) {
		arg1 = assign(eval(arg1));
		ret = getAtomIntVal(arg1, &val1);
		release(arg1);
		if (!ret) {
			return(eval_bad_arg);
		}
		arg2 = assign(eval(arg2));
		ret = getAtomIntVal(arg2, &val2);
		release(arg2);
		if (!ret) {
			return(eval_bad_arg);
		}
		*resultPP = mkintatom(val1+val2);
	}
	return(get);
}

static evalOkE
eval_assq(Lptr Sexpr, Lptr *resultPP) {
	Lptr	arg1, arg2;
	evalOkE	get;

	get = args2(Sexpr, &arg1, &arg2);
	if (get == eval_ok) {
		arg1 = assign(eval(arg1));
		arg2 = assign(eval(arg2));
		*resultPP = assq(arg1, arg2);
		release(arg1);
		release(arg2);
	}
	return(get);
}

static evalOkE
eval_atom(Lptr Sexpr, Lptr *resultPP) {
	Lptr	arg1, result;
	evalOkE	get;

	get = args1(Sexpr, &arg1);
	if (get == eval_ok) {
		arg1 = assign(eval(arg1));
		if (vAtom(arg1)) {
			result = getTrue();
		} else {
			result = getNil();
		}
		release(arg1);
		*resultPP = result;
	}
	return(get);
}

static evalOkE
eval_beginver(Lptr Sexpr, Lptr *resultPP)
{
	evalOkE	get;

	get = args0(Sexpr);
	if (get == eval_ok) {
		beginVer();
		*resultPP = mkintatom(getCurrentVersion());
	}
	return(get);
}

static evalOkE
eval_commitver(Lptr Sexpr, Lptr *resultPP)
{
	evalOkE	get;

	get = args0(Sexpr);
	if (get == eval_ok) {
		if (!getCurrentVersion()) {
			return(eval_cant_abort);
		}
		commitVer();
		*resultPP = mkintatom(getCurrentVersion());
	}
	return(get);
}

static evalOkE
eval_cons(Lptr Sexpr, Lptr *resultPP)
{
	Lptr	arg1, arg2;
	evalOkE	get;

	get = args2(Sexpr, &arg1, &arg2);
	if (get == eval_ok) {
		arg1 = assign(eval(arg1));
		arg2 = assign(eval(arg2));
		*resultPP = cons(arg1, arg2);
		release(arg1);
		release(arg2);
	}
	return(get);
}



static evalOkE
eval_divide(Lptr Sexpr, Lptr *resultPP){
	Lptr arg1, arg2;
	evalOkE	get;
	int ret;
	int val1, val2;

	get = args2(Sexpr, &arg1, &arg2);
	if (get == eval_ok) {
		arg1 = assign(eval(arg1));
		ret = getAtomIntVal(arg1, &val1);
		release(arg1);
		if (!ret) {
			return(eval_bad_arg);
		}
		arg2 = assign(eval(arg2));
		ret = getAtomIntVal(arg2, &val2);
		release(arg2);
		if (!ret ||!val2) {
			return(eval_bad_arg);
		}
		*resultPP = mkintatom(val1/val2);
	}
	return(get);
}

static evalOkE
eval_eq(Lptr Sexpr, Lptr *resultPP) {
	Lptr	arg1, arg2, result;
	evalOkE	get;

	get = args2(Sexpr, &arg1, &arg2);
	if (get == eval_ok) {
		arg1 = assign(eval(arg1));
		arg2 = assign(eval(arg2));

		if (eq(arg1, arg2)) {
			result = getTrue();
		} else {
			result = getNil();
		}
		release(arg1);
		release(arg2);
		*resultPP = result;
	}
	return(get);
}

static evalOkE
eval_eval(Lptr Sexpr, Lptr *resultPP)
{
	Lptr	arg1, result;
	evalOkE	get;

	get = args1(Sexpr, &arg1);
	if (get == eval_ok) {
		result	  = eval(arg1);
		*resultPP = eval(result);
	}
	return(get);
}

static evalOkE 
eval_findListHead(Lptr Sexpr, Lptr *resultPP) {
	Lptr	arg1, arg2;
	evalOkE	get;

	get = args2(Sexpr, &arg1, &arg2);
	if (get == eval_ok) {
		*resultPP = findListHead(arg1, arg2);
	}
	return(get);
}

static evalOkE
eval_firstatom(Lptr Sexpr, Lptr *resultPP)
{
	evalOkE	get;

	get = args0(Sexpr);
	if (get == eval_ok) {
		*resultPP = firstatom();
	}
	return(get);
}

static evalOkE 
eval_getAttributesList(Lptr Sexpr, Lptr *resultPP) {
	Lptr	arg1, arg2;
	evalOkE	get;

	get = args2(Sexpr, &arg1, &arg2);
	if (get == eval_ok) {
		*resultPP = findListHead(arg1, arg2);
	}
	return(get);
}

static evalOkE
eval_getprop(Lptr Sexpr, Lptr *resultPP) {
	Lptr	arg1, arg2;
	evalOkE	get;

	get = args2(Sexpr, &arg1, &arg2);
	if (get == eval_ok) {
		arg1 = assign(eval(arg1));
		if (Vtype(arg1) != STRTYPE) {
			get = eval_not_proplist;
		} else {
			arg2 = assign(eval(arg2));
			*resultPP = getprop(arg1, arg2);
			release(arg2);
		}
		release(arg1);
	}
	return(get);
}

static evalOkE
eval_intp(Lptr Sexpr, Lptr *resultPP) {
	Lptr	arg1, result;
	evalOkE	get;

	get = args1(Sexpr, &arg1);
	if (get == eval_ok) {
		arg1 = assign(eval(arg1));
		if (Vtype(arg1) == INTTYPE) {
			result = getTrue();
		} else {
			result = getNil();
		}
		release(arg1);
		*resultPP = result;
	}
	return(get);
}
	
static evalOkE
eval_listp(Lptr Sexpr, Lptr *resultPP) {
	Lptr	arg1, result;
	evalOkE	get;

	get = args1(Sexpr, &arg1);
	if (get == eval_ok) {
		arg1 = assign(eval(arg1));
		if (vListp(arg1)) {
			result = getTrue();
		} else {
			result = getNil();
		}
		release(arg1);
		*resultPP = result;
	}
	return(get);
}

static evalOkE
eval_irplaca(Lptr Sexpr, Lptr *resultPP) {
	Lptr	arg1, arg2;
	evalOkE	get;

	get = args2(Sexpr, &arg1, &arg2);
	if (get == eval_ok) {
		arg1 = assign(eval(arg1));
		if (Vtype(arg1) != CONSTYPE) {
			release(arg1);
			return(eval_bad_arg);
		}
		arg2 = assign(eval(arg2));
		irplaca(arg1, arg2);
		release(arg2);
		*resultPP = decrement(arg1);
	}
	return(get);
}

static evalOkE
eval_irplacd(Lptr Sexpr, Lptr *resultPP) {
	Lptr	arg1, arg2;
	evalOkE	get;

	get = args2(Sexpr, &arg1, &arg2);
	if (get == eval_ok) {
		arg1 = assign(eval(arg1));
		if (Vtype(arg1) != CONSTYPE) {
			release(arg1);
			return(eval_bad_arg);
		}
		arg2 = assign(eval(arg2));
		irplacd(arg1, arg2);
		release(arg2);
		*resultPP = decrement(arg1);
	}
	return(get);
}

static evalOkE 
eval_listUnion(Lptr Sexpr, Lptr *resultPP) {
	Lptr	arg1, arg2;
	evalOkE	get;

	get = args2(Sexpr, &arg1, &arg2);
	if (get == eval_ok) {
		*resultPP = listUnion(arg1, arg2);
	}
	return(get);
}

static evalOkE
eval_minus(Lptr Sexpr, Lptr *resultPP){
	Lptr arg1, arg2;
	int val1, val2;
	int ret;
	evalOkE	get;

	get = args1_2(Sexpr, &arg1, &arg2);
	if (get == eval_ok) {
		arg1 = assign(eval(arg1));
		ret = getAtomIntVal(arg1, &val1);
		release(arg1);
		if (!ret) {
			return(eval_bad_arg);
		}
		if(!arg2){
			*resultPP = mkintatom(-val1); 
		} else {
			arg2 = assign(eval(arg2));
			ret = getAtomIntVal(arg2, &val2);
			release(arg2);
			if(!ret){
				return(eval_bad_arg);
			}
			*resultPP = mkintatom(val1-val2);
	}	}
	return(get);
}

static evalOkE
eval_mod(Lptr Sexpr, Lptr *resultPP){
	Lptr arg1, arg2;
	int ret;
	int val1, val2;
	evalOkE	get;

	get = args2(Sexpr, &arg1, &arg2);
	if (get == eval_ok) {
		arg1 = assign(eval(arg1));
		ret = getAtomIntVal(arg1, &val1);
		release(arg1);
		if (!ret) {
			return(eval_bad_arg);
		}
		arg2 = assign(eval(arg2));
		ret = getAtomIntVal(arg2, &val2);
		release(arg2);
		if (!ret ||!val2) {
			return(eval_bad_arg);
		}
		*resultPP = mkintatom(val1 % val2);
	}
	return(get);
}

static evalOkE
eval_neq(Lptr Sexpr, Lptr *resultPP)
{
	Lptr	arg1, arg2, result;
	evalOkE	get;

	get = args2(Sexpr, &arg1, &arg2);
	if (get == eval_ok) {
		 arg1 = assign(eval(arg1)); /* added by Henry */
		 arg2 = assign(eval(arg2)); /* added by Henry */

		if (neq(arg1, arg2)) {
			result = getTrue();
		} else {
			result = getNil();
		}
		*resultPP = result;
		release(arg1);
		release(arg2);
	}
	return(get);
}

static evalOkE
eval_nextatom(Lptr Sexpr, Lptr *resultPP) {
	Lptr	arg1;
	evalOkE	get;

	get = args1(Sexpr, &arg1);
	if (get == eval_ok) {
		arg1 = assign(eval(arg1));
   
		if (Vtype(arg1) == STRTYPE) {
			*resultPP = nextatom(arg1);
		} else {
			get = eval_bad_arg;
		}
		release(arg1);
	}
	return(get);
}

static evalOkE
eval_null(Lptr Sexpr, Lptr *resultPP) {
	Lptr	arg1, result;
	evalOkE	get;

	get = args1(Sexpr, &arg1);
	if (get == eval_ok) {
		arg1 = assign(eval(arg1));
		if (vNull(arg1)) {
			result = getTrue();
		} else {
			result = getNil();
		}
		release(arg1);
		*resultPP = result;
	}
	return(get);
}
   
static evalOkE
eval_pputprop(Lptr Sexpr, Lptr *resultPP) {
	Lptr	arg1, arg2, arg3;
	evalOkE	get;

	get = args3(Sexpr, &arg1, &arg2, &arg3);
	if (get == eval_ok) {
		arg1 = assign(eval(arg1));
		if (Vtype(arg1) != STRTYPE) {
			get = eval_not_proplist;
		} else {
			arg2 = assign(eval(arg2));
			arg3 = assign(eval(arg3));

			*resultPP = pputprop(arg1, arg2, arg3);
			release(arg2);
			release(arg3);
		}
		release(arg1);
	}
	return(get);
}

static evalOkE
eval_putprop(Lptr Sexpr, Lptr *resultPP) {
	Lptr	arg1, arg2, arg3;
	evalOkE	get;

	get = args3(Sexpr, &arg1, &arg2, &arg3);
	if (get == eval_ok) {
		arg1 = assign(eval(arg1));
		if (Vtype(arg1) != STRTYPE) {
			get = eval_not_proplist;
		} else {
			arg2 = assign(eval(arg2));
			arg3 = assign(eval(arg3));

			*resultPP = putprop(arg1, arg2, arg3);
			release(arg2);
			release(arg3);
		}
		release(arg1);
	}
	return(get);
}

static evalOkE
eval_quote(Lptr Sexpr, Lptr *resultPP)
{
	return(args1(Sexpr, resultPP));
}

static evalOkE
eval_read(Lptr Sexpr, Lptr *resultPP)
{
	Lptr	arg1, frag, result;
	char	*stringP;
	int 	lth;
	evalOkE	get;

	get = args1(Sexpr, &arg1);
	if (get == eval_ok) {
		arg1 = assign(eval(arg1));
		switch (Vtype(arg1)) {
		case STRTYPE:
			frag = Next_frag(arg1);
			lth  = stringLth(frag);
			stringP = _alloca(lth + 1);
			if (!stringP) {
				return(eval_outofmem);
			}
			strStoreToString(frag, stringP);
			break;
		case INTTYPE:
			stringP = _alloca(20);
			if (!stringP) {
				return(eval_outofmem);
			}
			sprintf(stringP, "%d", Intval(arg1));
			break;
		default:
			release(arg1);
			return(eval_bad_arg);
		}
		release(arg1);
		result = lread(stringP);
		if (result == NULL) {
			return(eval_bad_arg);
		}
		*resultPP = result;
	}
	return(get);
}

static evalOkE
eval_remob(Lptr Sexpr, Lptr *resultPP) {
	Lptr	arg1;
	evalOkE	get;

	get = args1(Sexpr, &arg1);
	if (get == eval_ok) {
		arg1 = assign(eval(arg1));
		switch (Vtype(arg1)) {
			case STRTYPE:
				remob(arg1);
				break;
			default:
				release(arg1);
				return(eval_bad_arg);
		}
		release(arg1);
		*resultPP = getNil();
	}
	return(eval_ok);
}

static evalOkE
eval_set(Lptr Sexpr, Lptr *resultPP) {
	Lptr	arg1, arg2;
	evalOkE	get;

	get = args2(Sexpr, &arg1, &arg2);
	if (get == eval_ok) {
		arg1 = assign(eval(arg1));
		if (Vtype(arg1) != STRTYPE) {
			release(arg1);
			return(eval_bad_arg);
		}
		arg2 = assign(eval(arg2));
		setdef(arg1, arg2);
		release(arg1);
		*resultPP = decrement(arg2);
	}
	return(get);
}

static evalOkE
eval_setq(Lptr Sexpr, Lptr *resultPP) {
	Lptr	arg1, arg2;
	evalOkE	get;

	get = args2(Sexpr, &arg1, &arg2);
	if (get == eval_ok) {
		if (Vtype(arg1) != STRTYPE) {
			return(eval_bad_arg);
		}
		arg1 = assign(arg1);
		arg2 = assign(eval(arg2));
		setdef(arg1, arg2);
		release(arg1);
		*resultPP = decrement(arg2);
	}
	return(get);
}

static evalOkE
eval_times(Lptr Sexpr, Lptr *resultPP) {
	Lptr	arg1, arg2;
	int 	val1, val2;
	int 	ret;
	evalOkE	get;

	get = args2(Sexpr, &arg1, &arg2);
	if (get == eval_ok) {
		arg1 = assign(eval(arg1));
		ret = getAtomIntVal(arg1, &val1);
		release(arg1);
		if (!ret) {
			return(eval_bad_arg);
		}

		arg2 = assign(eval(arg2));
		ret = getAtomIntVal(arg2, &val2);
		release(arg2);
		if (!ret) {
			return(eval_bad_arg);
		}
		*resultPP = mkintatom(val1*val2);
	}
	return(get);
}

static evalOkE
eval_rplaca(Lptr Sexpr, Lptr *resultPP) {
	Lptr	arg1, arg2;
	evalOkE	get;

	get = args2(Sexpr, &arg1, &arg2);
	if (get == eval_ok) {
		arg1 = assign(eval(arg1));
		if (Vtype(arg1) != CONSTYPE) {
			release(arg1);
			return(eval_bad_arg);
		}
		arg2 = assign(eval(arg2));
		rplaca(arg1, arg2);
		release(arg2);

		*resultPP = decrement(arg1);
	}
	return(get);
}

static evalOkE
eval_rplacd(Lptr Sexpr, Lptr *resultPP) {
	Lptr	arg1, arg2;
	evalOkE	get;

	get = args2(Sexpr, &arg1, &arg2);
	if (get == eval_ok) {
		arg1 = assign(eval(arg1));
		if (Vtype(arg1) != CONSTYPE) {
			release(arg1);
			return(eval_bad_arg);
		}
		arg2 = assign(eval(arg2));
		rplacd(arg1, arg2);
		release(arg2);

		*resultPP = decrement(arg1);
	}
	return(get);
}

/**************************************************************************
Description: 
  evaluate the expression, which can be math expression or plan 
  algebra expression

Input parameter: 
  Sexpr: the list that represents the expression
 
Returns:
  The Lptr that represents the result 

Note:
  Lptr can be atoms, i.e. string or integer
  Complex expressions can be: add, assq, bindq, concept-subsumes?,
  concept-ancestors, concept-descendants, distinctL, enforceNewQ,
  equivalent, eval, findListHead, getAttributesList, implies, 
  in-knowledge-base, listUnion, multiply, neq, quote		
**************************************************************************/


/*	 N.B. **This result must subsequently be released** */

static int		g_function_cnt	= 0;
static int		g_max_functions = 0;
static evalOkE (**g_functionsP)(Lptr, Lptr *) = 0;

void addfunc(char *nameP, evalOkE (*fun)(Lptr, Lptr *)) 
{
	Sptr	sptr, func;

	if (g_function_cnt == g_max_functions) {
		if (!g_function_cnt) {
			g_max_functions = 128;
			g_functionsP = malloc(sizeof(*g_functionsP) * g_max_functions);
		} else {
			g_max_functions <<= 1;
			g_functionsP = realloc(g_functionsP, sizeof(*g_functionsP) * g_max_functions);
		}
		if (!g_functionsP) {
			fprintf(stderr, "Unable to grow g_functionsP\n");
			exit(1);
	}	}
	g_functionsP[g_function_cnt] = fun;

	sptr = mkstringatom(nameP);
	func = assign(mkstringatom("func"));
	putprop(sptr, func, mkintatom(g_function_cnt++));
	release(func);
}

void inittlisp(void) 
{
	typedef struct {
		char	*nameP;
		evalOkE (*fun)(Lptr, Lptr *);
	} keywordT;

	static const keywordT entry[] =
	{
		"abortver", 			eval_abortver,
		"add",					eval_add,
		"atom", 				eval_atom,
		"assq", 				eval_assq,
		"beginver", 			eval_beginver,
		"commitver",			eval_commitver,
		"cons", 				eval_cons,
		"divide",				eval_divide,
		"eq",					eval_eq,
		"eval", 				eval_eval,
		"findListHead", 		eval_findListHead,
		"firstatom",			eval_firstatom,
		"getAttributesList",	eval_getAttributesList,
		"getprop",				eval_getprop,
		"intp", 				eval_intp,
		"irplaca",				eval_irplaca,
		"irplacd",				eval_irplacd,
		"listUnion",			eval_listUnion,
		"listp",				eval_listp,
		"minus",				eval_minus,
		"mod",					eval_mod,
		"neq",					eval_neq,
		"nextatom", 			eval_nextatom,
		"null", 				eval_null,
		"pputprop",				eval_pputprop,
		"putprop",				eval_putprop,
		"quote",				eval_quote,
		"read", 				eval_read,
		"remob",				eval_remob,
		"rplaca",				eval_rplaca,
		"rplacd",				eval_rplacd,
		"set",					eval_set,
		"setq", 				eval_setq,
		"times",				eval_times
	};

	int			i;

	registerVtable(&nil_vtable, 0);
	registerVtable(&undef_vtable, 0);
	registerVtable(&error_vtable, 0);
	registerVtable(&cons_vtable, &g_memmgr);
	registerVtable(&cons_vtable_marked, &g_memmgr);
	registerVtable(&frag_vtable, &g_memmgr);
	registerVtable(&int_vtable, &g_memmgr);
	registerVtable(&int_vtable_marked, &g_memmgr);
	registerVtable(&str_vtable, &g_memmgr);
	registerVtable(&str_vtable_marked, &g_memmgr);

	for (i = 0; i < sizeof(entry)/sizeof(*entry); ++i) { 
		addfunc(entry[i].nameP, entry[i].fun);
}	}

Lptr eval(Lptr Sexpr) 
{
	static const char *eval_error[] = {
		"Ok",
		"Illegal argument to eval",
		"Illegal function name",
		"Too few arguments",
		"Too many arguments",
		"Bad argument",
		"Out of stack space",
		"Undefined",
		"No atom has this name",
		"No func indicator",
		"Func indicator not int",
		"Func indicator out of range",
		"Can't abort/commit when at version 0",
		"First argument not of type that can have a property list"

	};

	char			*nameP;
	evalOkE 		ret;

	//printf("eval "); printS(Sexpr); printf("\n");
	
	nameP = 0;
	switch (Vtype(Sexpr)) {
	case INTTYPE:
		return(Sexpr);
	case STRTYPE:
		return (getdef(Sexpr));
	case CONSTYPE:
		break;
	default:
		ret = eval_illegal_arg;
		goto fail;
	}
	
	{
		Sptr	fnameStore;
		Sptr	atom, func;
		Fptr	frag;
		Iptr	value;
		char	*name1P;
		int		lth, c, i;
		Lptr	result;


		fnameStore = assign(car(Sexpr));
		switch (Vtype(fnameStore)) {
		case STRTYPE:
			break;
		case CONSTYPE:
			atom = assign(eval(fnameStore));
			release(fnameStore);
			if (Vtype(atom) == STRTYPE) {
				fnameStore = atom;
				break;
			}
			release(atom);
		default:
			ret = eval_illegal_name;
			release(fnameStore);
			goto fail;
		}

		frag	= Start_frag(fnameStore);
		lth 	= stringLth(frag);
		nameP	= _alloca(lth + 1);
		if (!nameP) {
			ret = eval_outofmem;
			release(fnameStore);
			goto fail;
		}
		strStoreToString(frag, nameP);
		release(fnameStore);
	
		c = nameP[0];
		if (c != 'i') {
			name1P = nameP;
		} else {
			name1P = nameP + 1;
			c	   = *name1P;
			--lth;
		}
		if (c == 'c' && lth > 2 && name1P[lth-1] == 'r') {

			for (i = lth-2; i > 0; --i) {
				if (name1P[i] != 'a' && name1P[i] != 'd') {
					break;
			}	}
			if (!i) {
				Lptr	arg, eval1;
				
				arg = cdr(Sexpr);
				switch (Vtype(arg)) {
				case CONSTYPE:
					break;
				case NILTYPE:
					ret = eval_too_few;
					goto fail;
				default:
					ret = eval_bad_arg;
					goto fail;
				}
				eval1 = assign(eval(car(arg)));
				if (name1P == nameP) {
					result = cvr(name1P+1, eval1);
				} else {
					result = icvr(name1P+1, eval1);
				}
				release(eval1);
				return(result);
		}	 }
			
		atom = findhashstr(nameP);
		if (!atom) {
			ret = eval_func_missing;
			goto fail;
		}
		func  = assign(mkstringatom("func"));
		value = getprop(atom, func);
		switch (Vtype(value)) {
		case NILTYPE:
		case UNDEFTYPE:
			ret = eval_func_not_func;
			break;
		case INTTYPE:
			i = Intval(value);
			if (i < 0 || i >= g_function_cnt) {
				ret = eval_func_range_err;
				break;
			}
			ret = g_functionsP[i](cdr(Sexpr), &result);
			break;
		default:
			ret = eval_func_not_int;
		}
		release(func);

		if (ret == eval_ok){
			return(result);
	}	}
fail:
	if (!nameP) {
		nameP = "<<UNKNOWN>>";
	}
	fprintf(stderr,"%s: %s\n", nameP, eval_error[ret]);
	return(getError());
}
 
Lptr vaddList (Lptr ListHead, Lptr Node) {
	return(rplaca(ListHead,cons(Node,car(ListHead))));
}

/* This handles reference counting if one wants to test vNull(eval(Sexpr))
 * Unlike vNull(eval(Sexpr)) this releases the value evaluated
 */

boolean nullEval(Lptr Sexpr)
{
	Lptr	result;
	boolean ret;

	result = assign(eval(Sexpr));
	ret    = vNull(result);
	release(result);
	return(ret);
}




