/***************************************************************************
								  pattern.c  
							 -------------------
	Description 		  C file for Pattern Matcher/Rule Controller
 ***************************************************************************/
/***************************************************************************
 *																		   *
 *	 This program is free software; you can redistribute it and/or modify  *
 *	 it under the terms of the GNU General Public License as published by  *
 *	 the Free Software Foundation; either version 2 of the License, or	   *
 *	 (at your option) any later version.								   *
 *																		   *
 ***************************************************************************/

#include "config.h"
#include <stdio.h>

#include "memmgr.h"
#include "tlisp.h"

/****************************************************************************
 Pattern Matcher
 ****************************************************************************/

Lptr matchVar=NULL, matchVarL=NULL, freeMatchVars=NULL, matchVarStack=NULL;

void initVars(void) {

	Lptr nilset;
	nilset=getNil();
	
	if (matchVar == NULL) {
		setSptr(&matchVar, nilset);
		setSptr(&matchVarL, cons(nilset,nilset));
		setSptr(&freeMatchVars,nilset);
		setSptr(&matchVarStack,nilset);
	}
}

void finalizeVars(void) {
	setSptr(&matchVarL, NULL);
	setSptr(&matchVar, NULL);
	setSptr(&freeMatchVars, NULL);
	setSptr(&matchVarStack, NULL);
}

Lptr topCopy(Lptr L) {
	if (vNull(L))
		return (getNil());
	else
		return ( cons( car(L), topCopy(cdr(L)) ) );
}

Lptr getBindVal(Lptr var) {
	return (getprop(var,matchVar));
}

Lptr putBindVal(Lptr var, Lptr val) {
	return (putprop(var, matchVar, val));
}

Lptr pushMatchVar(void) {

	setSptr(&matchVarStack, rplacd(matchVarL,matchVarStack));
	
	if (vNull(freeMatchVars)) {
		setSptr(&matchVar, gensym("P"));
		setSptr(&matchVarL, cons(matchVar,getNil()));
	} else {
		setSptr(&matchVar, car(freeMatchVars));
		setSptr(&matchVarL, freeMatchVars);
		setSptr(&freeMatchVars, cdr(freeMatchVars));
		rplacd(matchVarL,getNil());
	}
	return (freeMatchVars);
}

Lptr popMatchVar(void) {

	if (vNull(matchVarStack))
		printf("vNull match var stack in pop operation\n");
	else {
		setSptr(&freeMatchVars, rplacd(matchVarL, freeMatchVars));

		setSptr(&matchVar, car(matchVarStack));
		setSptr(&matchVarL, matchVarStack);
		setSptr(&matchVarStack, cdr(matchVarStack));
		rplacd(matchVarL,getNil());
	}
	return (getNil());
}

Lptr Build(Lptr P) {
	if (vAtom(P) || vNull(P))
		return (P);
	else if (eql(car(P),"<"))
		return (cons(getBindVal(car(cdr(P))), Build(cdr(cdr(P)))) );
	else if (eql(car(P),"<q"))
		return (cons(getBindVal(car(cdr(P))), Build(cdr(cdr(P)))) ); // KWOTE?
	else if (eql(car(P),"<<"))
		return (append(getBindVal(car(cdr(P))), Build(cdr(cdr(P)))) );
	else if (eql(car(P),"!"))
		return (cons(car(cdr(P)), Build(cdr(cdr(P)))) );
	else
		return (cons(Build(car(P)), Build(cdr(P))) );
}

Lptr nlist(Lptr In) {
	return(cons(In,getNil()));
}

boolean Match(Lptr Pat, Lptr L) {
	Lptr topL, nPat;
	boolean rMatch;
	//printf("\n\nMatch ");printS(0, Pat);printf("\n To ");printS(0, L);printf("\n");

	if (!vListp(Pat))
		return (FALSE);

	if (!vListp(L))
		return (FALSE);

	if (vNull(Pat))
		return (vNull(L));

	if (vListp(car(Pat)))
		return (vListp(car(L)) && Match(car(Pat),car(L)) && Match (cdr(Pat),cdr(L)));

	if (eql(car(Pat),"*")) {
		if (vNull(cdr(Pat)))
			return TRUE;
		else if (Match(cdr(Pat),L))
			return TRUE;
		else {
			nPat = assign(	cons(mkstringatom("+"),cdr(Pat))  );
			rMatch = Match(nPat,L);
			release(nPat);
			return rMatch;
		}
	}

	if (eql(car(Pat),">*")) {
		if ( vNull(cdr(cdr(Pat))) ){
			topL=assign(  topCopy(L)  );
			putBindVal(car(cdr(Pat)),topL);
			release(topL);
			return TRUE;
		}
		else if ( eql(car(cdr(cdr(Pat))),"where") && vNull(cdr(cdr(cdr(cdr(Pat))))) ) {
			topL = assign(	topCopy(L)	);
			putBindVal(car(cdr(Pat)),topL);
			release(topL);
			return(Match(cdr(cdr(Pat)),getNil()));
		}
		else {
			boolean rMatch;
			nPat = assign(	cons(mkstringatom(">++"),cdr(Pat))	);
			putBindVal(car(cdr(Pat)),getNil());
			rMatch = Match(nPat,L);
			release(nPat);
			return (rMatch);
		}

	}

	if (eql(car(Pat),"<<")) {
		nPat = assign(	append(getBindVal(car(cdr(Pat))),cdr(cdr(Pat)))  );
		rMatch = Match(nPat,L);
		release(nPat);
		return (rMatch);
	}

	if (eql(car(Pat),"<")) {
		nPat = assign(	cons(getBindVal(car(cdr(Pat))),cdr(cdr(Pat)))  );
		rMatch = Match(nPat,L);
		release(nPat);
		return (rMatch);
	}

	if (eql(car(Pat),"<>")) {
		Lptr tmpPat;
		tmpPat = assign(append(car(cdr(Pat)),car(cdr(cdr(Pat)))));
		nPat = assign(append(tmpPat,cdr(cdr(cdr(Pat)))));
		rMatch = Match(nPat,L);
		release(tmpPat);
		release(nPat);
		if (rMatch)
			return (rMatch);
		tmpPat = assign(append(car(cdr(cdr(Pat))),car(cdr(Pat))));
		nPat = assign(append(tmpPat,cdr(cdr(cdr(Pat)))));
		rMatch = Match(nPat,L);
		release(tmpPat);
		release(nPat);
		return (rMatch);
	}

	if (eql(car(Pat),"where")) {
		Lptr buildres, evalres;
		int doMatch;

		buildres = assign(Build(car(cdr(Pat))));
		evalres = assign(eval(buildres));
		doMatch = !vNull(evalres);
		release(buildres);
		release(evalres);

		if ( doMatch )
			return (Match(cdr(cdr(Pat)),L));
		else
			return (FALSE);
	}

	if (eql(car(Pat),">++")) {
		if (vNull(L))
			return (Match(cdr(cdr(Pat)),L));
		else if (Match(cdr(cdr(Pat)),L))
			return (TRUE);
		else {
			Lptr BVal = getBindVal(car(cdr(Pat)));
			Lptr EList = nlist(car(L)); // list (car(L))
			if (vNull(BVal))
				putBindVal(car(cdr(Pat)), EList);
			else
				rplacd(last(BVal),EList);
			return (Match(Pat,cdr(L)));
		}
	}

	if (vNull(L))
		return (FALSE);

	if (eql(car(Pat),"!"))
		return ( eq(car(cdr(Pat)),car(L)) && Match(cdr(cdr(Pat)),cdr(L)) );

	if (eql(car(Pat),"?"))
		return (Match(cdr(Pat),cdr(L)));

	if (eql(car(Pat),"+")) {
		if (Match(cdr(Pat),cdr(L)))
			return TRUE;
		else
			return (Match(Pat,cdr(L)));
	}

	if (eql(car(Pat),">")) {
//		lseditT lsedit = {"..", 0, 0, 0};
		putBindVal(car(cdr(Pat)),car(L));
//		dump_lsedit(&lsedit, Pat);
//		dump_lsedit(&lsedit, L);
		return (Match(cdr(cdr(Pat)),cdr(L)));
	}

	if (eql(car(Pat),">+")) {
		putBindVal(car(cdr(Pat)),nlist(car(L)));
		nPat = assign(cons(mkstringatom(">++"),cdr(Pat)));
		rMatch =Match(nPat,cdr(L));
		release(nPat);
		return (rMatch);
	}

	if (eql(car(Pat),"or")) {
		if (member(car(L),car(cdr(Pat))))
			return (Match(cdr(cdr(Pat)),cdr(L)));
		else
			return (FALSE);
	}

	if (eql(car(Pat),">or")) {
		if (member(car(L),car(cdr(Pat)))){
			putBindVal(car(cdr(cdr(Pat))),car(L));
			return (Match(cdr(cdr(cdr(Pat))),cdr(L)));
		}
		else
			return (FALSE);
	}

	if (eql(car(Pat),">>or")) {
		Lptr tmpOrL, var;

		var =car(cdr(cdr(Pat)));
		for (tmpOrL =car(cdr(Pat)); !vNull(tmpOrL) ; tmpOrL = cdr(tmpOrL)) {
			nPat = assign(append(car(tmpOrL),cdr(cdr(cdr(Pat)))));
			//printS(0, nPat); printS(0, L); printf("\n");
			rMatch = Match(nPat,L);
			release(nPat);
			if (rMatch) {
				putBindVal(var, car(L));
				return (rMatch);
			}
		}
		return (FALSE);
	}	 
	
	return (eq(car(Pat),car(L)) && Match(cdr(Pat),cdr(L)));
}

Lptr Bindq(Lptr BList) {
	Lptr blist, Fn;
	for (blist=BList, Fn=getNil(); ; blist=cdr(cdr(blist))) {
		if (vNull(blist))
			return (Fn);
		else {
			Fn=assign(eval(car(cdr(blist)))); 
			putBindVal(car(blist),Fn);
			release(Fn);
		}
	}
}

Lptr Bind(Lptr BList) {
	Lptr blist, Fn1, Fn2;
	for (blist=BList, Fn2=getNil(); ; blist=cdr(cdr(blist))) {
		if (vNull(blist))
			return (Fn2);
		else {
			Fn1 = assign(eval(car(blist)));
			Fn2 = assign(eval(car(cdr(blist)))); 
			putBindVal(Fn1,Fn2);
			release(Fn1);
			release(Fn2);
		}
	}
}

/************************************************************************
  Rule Based Rewriting
 ************************************************************************/


boolean ApplyRuleControl (Lptr Control, Lptr Form) {
	Lptr RuleList;
	boolean Result;
//	  printf("ApplyRule "); printS(0, Control); printf("\n To "); printS(0, Form); printf("\n\n");
	
	if (vAtom(Control)) {
		Lptr Rule;
		pushMatchVar();
		Rule = getprop(Control, mkstringatom("RWRule"));
		if (Match(car(Rule),Form)){
			Lptr NewForm,tmpB, eval1;
			for (tmpB=cdr(cdr(Rule)); !vNull(tmpB) ;tmpB=cdr(tmpB)) {
				Lptr BuildS;
				BuildS = assign(Build(car(tmpB)));
				eval1  = assign(eval(BuildS));
				release(eval1);
				release(BuildS);
			}
			NewForm = Build(car(cdr(Rule)));
			rplaca(Form,car(NewForm));
			rplacd(Form,cdr(NewForm));
			release(NewForm);
			popMatchVar();
			return (TRUE);
		}
		else
			return (!vNull( popMatchVar() )); // return false if pop nil?
	}

	if (eql(car(Control),"call")) {
		return ( ApplyRuleControl (getprop(car(cdr(Control)),mkstringatom("RuleControl")), Form) );
	}

	if (eql(car(Control),"not")) {
		return ( !ApplyRuleControl(car(cdr(Control)),Form) );
	}

	if (eql(car(Control),"or")) {
		for (RuleList=cdr(Control);;RuleList=cdr(RuleList)) {
			if (vNull(RuleList))
				return (FALSE);
			else
				if (ApplyRuleControl(car(RuleList),Form))
					return (TRUE);
		}
	}

	if (eql(car(Control),"and")) {
		for (RuleList=cdr(Control);;RuleList=cdr(RuleList)) {
			if (vNull(RuleList))
				return (TRUE);
			else
				if (!ApplyRuleControl(car(RuleList),Form))
					return (FALSE);
		}
	}

	if (eql(car(Control),"seq")) {
		for (RuleList = cdr(Control), Result = FALSE;;RuleList = cdr(RuleList)) {
			if (vNull(RuleList))
				return (Result);
			else
				if (ApplyRuleControl(car(RuleList),Form))
					Result = TRUE;
		}
	}

	if (eql(car(Control),"rep")) {
		for (Result = FALSE;;Result =  TRUE) {
			if (!(ApplyRuleControl(car(cdr(Control)),Form)))
					return (Result);
		}
	}

	if (eql(car(Control),"if")) {
		if (Match(car(cdr(Control)), Form))
			return (ApplyRuleControl(car(cdr(cdr(Control))),Form));
		else
			return (FALSE);
	}

	if (eql(car(Control),"env")) {
		Lptr evalparm, reval;
		boolean envbool;
		evalparm=cons(car(cdr(Control)),cons(cons(mkstringatom("quote"),cons(Form,getNil())),getNil()));
		reval = assign(eval(evalparm));
		envbool = ApplyRuleControl(car(cdr(cdr(Control))),reval);
		release(evalparm);
		release(reval);
		return (envbool);
	}

	if (eql(car(Control),"map")) {
		Lptr FormList;
		for (FormList = Form, Result = FALSE;;FormList = cdr(FormList)) {
			if (vNull(FormList))
				return (Result);
			else
				if (ApplyRuleControl(car(cdr(Control)),car(FormList)))
					Result = TRUE;
		}
	}

	return (FALSE); // No rule applied
}

Lptr LoadRules (Lptr RList) {
	Lptr Rule, tempR;
	for (tempR = RList; !vNull(tempR) ; tempR = cdr(tempR) ) {
		Rule = car(tempR);
		putprop(car(Rule), mkstringatom("RWRule"), cdr(Rule));
	}
	return (RList);
}

Lptr UnLoadRules (Lptr RList) {
	Lptr Rule, tempR;
	for (tempR = RList; !vNull(tempR) ; tempR = cdr(tempR) ) {
		Rule = car(tempR);
		remprop(car(Rule), mkstringatom("RWRule"));
	}
	return (RList);
}


Lptr LoadControl (Lptr Control) {
	return (putprop(car(Control), mkstringatom("RuleControl"), car(cdr(Control)) ));
}

Lptr UnLoadControl (Lptr Control) {
	return (remprop(car(Control), mkstringatom("RuleControl")));
}

Lptr initControl(char *rwRules, char *controlProg) {
	Lptr RuleControl, RList, Control;

	RuleControl = assign(cons(lread(rwRules), cons(lread(controlProg), getNil())));
	RList = car(RuleControl);
	Control =car(cdr(RuleControl));
	initVars();
	LoadRules(RList);
	LoadControl(Control);
	return (RuleControl);
}

Lptr finalizeControl(Lptr RuleControl) {
	Lptr RList, Control;

	if (RuleControl != NULL) {
		RList = car(RuleControl);
		Control =car(cdr(RuleControl));

		UnLoadRules(RList);
		UnLoadControl(Control);
		finalizeVars();
		release(RuleControl);
	}
	return (NULL);
}

static evalOkE
eval_bindq(Lptr Sexpr, Lptr *resultPP) {

	Lptr	arg1;
	evalOkE	get;

	get = args1(Sexpr, &arg1);
	if (get == eval_ok) {
		*resultPP = Bindq(arg1);
	}
	return(get);
}

void initpattern(void) 
{
	typedef struct {
		char	*nameP;
		evalOkE (*fun)(Lptr, Lptr *);
	} keywordT;

	static const keywordT entry[] =
	{
		"bindq",				eval_bindq
	};

	int			i;

	for (i = 0; i < sizeof(entry)/sizeof(*entry); ++i) { 
		addfunc(entry[i].nameP, entry[i].fun);
}	}


