/***************************************************************************
 *                                  memmgr.c
 *                              -------------------
 * Description               C code for memory manager
 ****************************************************************************/
/***************************************************************************
 *  *                                                                                                                                                 *
 *  *       This program is free software; you can redistribute it and/or modify  *
 *  *       it under the terms of the GNU General Public License as published by  *
 *  *       the Free Software Foundation; either version 2 of the License, or     *
 *  *       (at your option) any later version.                                   *
 *  *                                                                             *
 ****************************************************************************/

#include "config.h"

#include <assert.h>
#include <malloc.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h> 


#include "memmgr.h"


/****************************************************************************
 Memory Allocation
*****************************************************************************/

// Linked list of all known vtableT classes
vtableT	*g_vtablesP = 0;
// Linked list of all known memMgrs.
static memMgrT *g_memMgrsP = 0;


/*
Returns one cell of memory
keeps the free lists
gets chunk from OS when no more free cells
*/

int is_true(void)
{
	return(1);
}

int is_false(void)
{
	return(0);
}

static int free_sizeof(void)
{
	return(sizeof(freeT));
}

static celltypeE free_stype(void)
{
	return(FREETYPE);
}

static const char * free_name(void)
{
	return("FREE");
}

static void free_dump(void *P, FILE *F)
{
	fprintf(F, "FREE");
}

static int	free_empty(void *fnode)
{
	fprintf(stderr, "Attempting to free %p more than once\n", fnode);
	return(0);
}

static void free_set_mark(void *);
static void free_clr_mark(void *);

static vtableT free_vtable =
{
	&free_sizeof,
	FREETYPE,
	&free_name,
	&free_dump,
	&free_empty,
	&free_set_mark,
	&free_clr_mark,
	&is_false
};

static vtableT free_vtable_marked =
{
	&free_sizeof,
	FREETYPE,
	&free_name,
	&free_dump,
	&free_empty,
	&free_set_mark,
	&free_clr_mark,
	&is_true
};

static void free_set_mark(void *fnode)
{
	Vtable((freeT *) fnode) = &free_vtable_marked;
}

static void free_clr_mark(void *fnode)
{
	Vtable((freeT *) fnode) = &free_vtable;
}

static void
initMemBlock(memMgrT *stateP,memBlocksT *blocksP, freeT	*chainP)
{
	char	*P, *nextP, *endP;
	int		entries, size;

	// Create a circular linked list

	entries = stateP->m_entries;
	size    = stateP->m_size;

	assert(entries && size);

	P       = blocksP->block;
	for (endP = P + entries * size; ; P = nextP) {
		Vtable(P)    = &free_vtable;
		nextP        = P + size;
		if (nextP >= endP) {
			break;
		}
		Next_free((freeT *) P) = (freeT *) nextP;
	}
	Next_free((freeT *) P)  = chainP;  // Close cyclic linked list
	stateP->m_prevP	   = (freeT *) P;  // Prev is what ever points to start of chain

	blocksP->nextP	   = stateP->m_headP;
	stateP->m_headP	   = blocksP;
}

freeT *
getMemBlock(vtableT *vtableP)
{
	memMgrT		*stateP;
	freeT		*prevP, *P;
	memBlocksT	*blocksP;

	stateP = vtableP->m_memmgrP;
	if (!stateP)
	{
		stateP = stateP;
	}
	
	assert(stateP);
/*
	static int cntr = 0;

	++cntr;
	if (cntr == 1448) {
		cntr = cntr;
	}
 */  
	prevP = stateP->m_prevP;
	if (!prevP) {
		blocksP = (memBlocksT *) malloc(sizeof(memBlocksT *) + (stateP->m_entries * stateP->m_size)); 
		if (!blocksP) { 
			fprintf(stderr, "Initial getMemBlock can't get memory\n");
			exit(1);
		}
		initMemBlock(stateP, blocksP, (freeT *) blocksP->block);
		prevP = stateP->m_prevP;
	}

	P = Next_free(prevP);

	// Increase blocks if down to last free block

	if (P == prevP) {
		blocksP = (memBlocksT *) malloc(sizeof(memBlocksT *) + (stateP->m_entries * stateP->m_size)); 
		if (!blocksP) { 
			fprintf(stderr, "Secondary getMemBlock can't get memory\n");
			exit(1);
		}
		// Add the new blocks into the chain
		Next_free(prevP) = (freeT *) blocksP->block;
		initMemBlock(stateP, blocksP, prevP);
		prevP = stateP->m_prevP;
		P = Next_free(prevP);
	}
	
	Next_free(prevP) = Next_free(P);
	assert(Vtable(P) == &free_vtable);

/*
	if (P == ((Lptr) 0x4619D0) && stype == CONSTYPE) {
		P = P;
	}
*/
	Vtable(P) = vtableP;
	return P;
}

/*
Return the cell to the free list
*/
void freeMemBlock(freeT *targetP) 
{
	freeT	*prevP;
	memMgrT	*memmgrP;

	memmgrP  = targetP->vtableP->m_memmgrP;
	assert(memmgrP);
	prevP    = memmgrP->m_prevP;
	assert(prevP);

/*
	if (fnode == stateP->m_headP->block+343) {
		fnode = fnode;
	}
*/
	Vtable(targetP)    = &free_vtable;
	Next_free(targetP) = Next_free(prevP);
	Next_free(prevP)   = targetP;
}

static void printFreeList(memMgrT *stateP) {
	int		i;
	freeT	*P, *prevP;
	
	i = 0;
	P = prevP = stateP->m_prevP;
	do {
		P = Next_free(P);
		assert(Vtable(P) == &free_vtable);
		printf("%i %p next %p\n", ++i, P, Next_free(P));
	} while (P != prevP);
}

void printFreeStats(memMgrT *stateP) {

	int 		i, j, k, max, size, entries;
	celltypeE	type;
	freeT		*freeP, *prevP;
	char		*P, *endP;
	memBlocksT	*headP, *tailP;
	int 		count[BADTYPE] = {0,0,0,0,0,0,0,0};
	const char *name[BADTYPE]  = {0,0,0,0,0,0,0,0}; 

	entries = stateP->m_entries;
	size    = stateP->m_size;

	printf("\nBlock Request size    = %i\n", entries);
	printf("\nIndividual size  e    = %i\n", size);

	prevP = stateP->m_prevP;

	if (!prevP) {
		return;
	}

	headP = stateP->m_headP;

	i = 1;
	for ( freeP = Next_free(prevP); freeP != prevP; freeP = Next_free(freeP)) {
		if (Vtable(freeP) != &free_vtable) {
			fprintf(stderr, "Cell %p on freelist has type %s\n", Vname(freeP)); 
			assert(0);
		}
		++i;
	}
	printf("Free cells in list    = %i\n", i);

	j	= 0;

	for (tailP = headP; tailP; tailP = tailP->nextP) {
		++j;
		P = tailP->block;
		for (endP = P + (entries * size); P < endP; P += size) {
			freeP = (freeT *) P;
			if (Vtable(freeP) != &free_vtable) {
				Vdump(freeP, stdout);
				fputc('\n', stdout);
			}
			type = Vtype(freeP);
			if (!count[type]++) {
				name[type] = Vname(P);
			}
	}	}
	
	j *= entries;
	printf("Total number of cells = %i\n", j);
	printf("Cells not freed       = %i\n\n", j- i);

	if (j != i) {
		max = sizeof(count)/sizeof(count[0]);
		for (k = 0; k < max; ++k) {
			if (count[k]) {
				printf(" Cell type %-10.10s = %i\n", name[k], count[k]);
		}	}
		printf("\n");

		// Make a good stab at identifying what it is that hasn't been freed
		// that is causing the overall problem.
		for (tailP = headP; tailP; tailP = tailP->nextP) {
			P = tailP->block;
			for (endP = P + (size * entries); P < endP; P += size) {
				freeP = (freeT *) P;
				if (Vtable(freeP) != &free_vtable) {
					Vempty(freeP);
		}	}	}

		for (tailP = headP; tailP; tailP = tailP->nextP) {
			P = tailP->block;
			for (endP = P + (size * entries); P < endP; P += size) { 
				freeP = (freeT *) P;
				if (Vtable(freeP) != &free_vtable) {
					printf("Villain %p ", freeP);
					Vdump(freeP, stdout);
					fputc('\n', stdout);
		}	}	}
	}

	if (i != count[FREETYPE]) {
		printf("**FREE LIST BROKEN**\n\n");
		printFreeList(stateP);
	}
}

void printAllFreeStats(void)
{
	memMgrT	*memMgrP;

	for (memMgrP = g_memMgrsP; memMgrP; memMgrP = memMgrP->m_nextP) {
		printFreeStats(memMgrP);
	}
}

void registerVtable(vtableT *vtableP, memMgrT *memmgrP)
{
	unsigned int	size;
	memMgrT			*knownP;

	// Can't register the same class more than once
	assert(!vtableP->m_nextP);
	// Can't switch memmgrs once specified.
	assert(!vtableP->m_memmgrP);

	size = vtableP->function_sizeof();
	if (memmgrP) {
		if (size < sizeof(freeT)) {
			size = sizeof(freeT);
		}
		if (size > memmgrP->m_size) {
			memmgrP->m_size = size;
		}
		vtableP->m_memmgrP = memmgrP;

		for (knownP = g_memMgrsP; knownP; knownP = knownP->m_nextP) {
			if (!knownP) {
				memmgrP->m_nextP = g_memMgrsP;
				g_memMgrsP       = memmgrP;
				break;
			}
			if (knownP == memmgrP) {
				break;
		}	}
	}

	vtableP->m_nextP = g_vtablesP;
	g_vtablesP       = vtableP;
}

void initmemmgr(void)
{
	registerVtable(&free_vtable, 0);
	registerVtable(&free_vtable_marked, 0);

	
	//initstructs();
}


