/***************************************************************************
								  main.c   
							 -------------------
	Description 		   Main Driving Function
 ***************************************************************************/
/***************************************************************************
 *																		   *
 *	 This program is free software; you can redistribute it and/or modify  *
 *	 it under the terms of the GNU General Public License as published by  *
 *	 the Free Software Foundation; either version 2 of the License, or	   *
 *	 (at your option) any later version.								   *
 *																		   *
 ***************************************************************************/

#include "config.h"

#include <sys/types.h>
#include <sys/timeb.h>
#include <time.h>
#include <malloc.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "memmgr.h"
#include "tlisp.h"
#include "pattern.h"
#include "cgu.h"
#include "reasoner.h"
#include "optimizer.h"
#include "sprint.h"
#include "lsedit.h"

#ifdef WIN32
typedef struct _timeb	timeT;
#else
typedef struct timeb	timeT;
#endif

static const char *test_name[] = {
	/* 0 */ "invalid",
	/* 1 */ "=",
	/* 2 */ "parse",
	/* 3 */ "expand1",
	/* 4 */ "expand2",
	/* 5 */ "eval",
	0
};

typedef enum {
	test_invalid	= 0,
	test_regression = 1,
	test_parse		= 2,
	test_expand1	= 3,
	test_expand2	= 4,
	test_eval		= 5
} testE;

static	testE	default_test_case = test_expand2;
static testE	test_case		  = test_expand2;

typedef struct {
	FILE	*F;
	char	*filenameP;
} stackT;

static	int 	elapse_flag    = 0; 		// Set to 1 if the time of execution should be reported
static	int 	eflag		   = 0; 		// Echo the input entered
static	char	*testdirP	   = 0; 		// The root directory in which test scripts are to be found 			
static	stackT	stack[128]; 				// Stack of open scripts
static	stackT	*stackP 	   = stack-1;	// Position currently at in the stack
static	int 	state		   = 0; 		// Expected function of the input 

static lseditT	lsedit = {0, 0, 0, 0};

static	sprintT actual;

static void
outofmemory(void)
{
	fprintf(stderr, "Out of memory\n");
	exit(1);
}

static testE
lookup_test(const char *nameP, int terminator)
{
	const char	*test_nameP;
	int 		i;
	unsigned int lth;

	for (i = 0; test_nameP = test_name[i]; ++i) {
		lth = (int) strlen(test_nameP);
		if (strncmp(nameP, test_nameP, lth)) {
			continue;
		}
		if (nameP[lth] != terminator) {
			continue;
		}
		return ((testE) i);
	}	
	return(test_invalid);
}

// Add the specified filename (stdin if null) to the stack of scripts

static int 
pushinput(const char *filenameP)
{
	FILE	*F;
	char	*fullfilenameP;
	int 	c, lth;

	state	  = 0;
	test_case = default_test_case;

	if (!filenameP) {
		F			  = stdin;
		fullfilenameP = 0;
	} else {
		for (; (c = *filenameP) == ' ' || c == '\t'; ++filenameP);
		if ((stackP < stack && !testdirP) || filenameP[0] == '/' || filenameP[0] == '\\' || strchr(filenameP, ':') ) {
simple: 	fullfilenameP = (char *) malloc(strlen(filenameP) + 1);
			if (!fullfilenameP) {
				outofmemory();
			}
			strcpy(fullfilenameP, filenameP);
		} else if (stackP < stack) {
usetestdir:
			lth = (int) strlen(testdirP);
			fullfilenameP = malloc(lth + strlen(filenameP) + 2);
			if (!fullfilenameP) {
				outofmemory();
			}
			strcpy(fullfilenameP, testdirP);
			fullfilenameP[lth] = '/';
			strcpy(fullfilenameP+lth+1, filenameP);
		} else {
			int 	lth;
			char	*dirP, *P, *P1;

			dirP = stackP->filenameP;
			if (!dirP) {
				goto usetestdir;
			}
			P	 = strrchr(dirP, '\\');
			P1	 = strrchr(dirP, '/');
			if (!P || (P1 && P1 > P)) {
				P = P1;
			} 
			if (!P) {
				goto simple;
			}
			fullfilenameP = malloc(P-dirP + strlen(filenameP) + 2);
			if (!fullfilenameP) {
				outofmemory();
			}
			lth = (int) ((P - dirP) + 1);
			strncpy(fullfilenameP, dirP, lth);
			strcpy(fullfilenameP+lth, filenameP);
		}
	
		if (!fullfilenameP) {
			F = 0;
		} else {
			F = fopen(fullfilenameP, "r");
		}
		if (!F) {
			fprintf(stderr, "Unable to open %s file '%s'\n", ((stackP < stack) ? "input" : "script"), fullfilenameP);
			free(fullfilenameP);
			return(0);
		}
		printf("\n***FILE*** %s\n", fullfilenameP);
	}
	++stackP;
	if ((stackP - stack) >= (sizeof(stack)/sizeof(*stack)) ) {
		fprintf(stderr, "to many @ redirections .. perhaps a cyclic relationship exists in scripts\n");
		exit(2);
	}
	
	stackP->F		  = F;
	stackP->filenameP = fullfilenameP;
	return(1);
}

// Unstack the set of scripts to be run

static int
popinput(void)
{
	state	  = 0;
	test_case = default_test_case;

	if (stackP->F != stdin) {
		fclose(stackP->F);
	}
	free(stackP->filenameP);
	--stackP;
	if (stackP < stack) {
		return(0);
	}
	return(1);
}

/* Read input from F into *input_bufPP.  Terminate at end of line if
 * terminator 0, else when the specified terminator character is encountered.
 *
 * Returns E_FAIL if F null, S_OK normally, S_FALSE on end of file or ^F
 */

static int
getinputcommand(FILE *F, int *sizeP, char **input_bufPP)
{
	char	*P;
	int 	c, c1, prev;
	int 	lth;
	int 	brackets;
	int 	quoted;
	int 	comment;
	int 	ret;

	if (!F) {
		fprintf(stderr, "Can't get command: input file null\n");
		exit(1);
	}

	if (!(P = *input_bufPP)) {
		P = malloc(256);
		*sizeP = 256;
		if (!P) {
			outofmemory();
		}
		*input_bufPP = P;
	}

	brackets   = 0;
	quoted	   = 0;
	comment    = 0;
	c1		   = 0;
	for (lth = 0;;) {
		prev = c1;
		c1	 = c = getc(F);
		if (feof(F)) {
			if (ferror(F)) {
				continue;
			}
			if (!lth) {
				return(0);
			}
			goto done1;
		}

		if (!lth) {
			switch (c) {
			case ' ':
			case '\t':
			case '\r':
			case '\n':
				continue;
			case '!':
			case '@':
			case '<':
				brackets = -1;
		}	}

		if (comment) {
			if (c == '\n') {
				comment = 0;
			}
			continue;
		}
		if (lth >= (int) *sizeP) {
			P = realloc(P, lth * lth);
			*sizeP = lth*lth;
			if (!P) {
				outofmemory();
			}
			*input_bufPP = P;
		}

		if (brackets >= 0) {
			switch (c) {
			case '(':
				if (!quoted) {
					++brackets;
				}
				break;
			case ')':
				if (!quoted) {
					--brackets;
				}
				break;
			case ';':	/* comment */
				if (!quoted) {
					comment = 1;
					continue;
				}
				break;
			case '"':
				if (prev != '\\') {
					++quoted;
				}
				break;
			case '\'':
				if (prev != '\\') {
					--quoted;
				}
				break;
			case '\\':
				if (prev == '\\') {
					c1 = 0;
				}
				break;
		}	}

		if (!quoted) {
			switch (c) {
			case 13:
				// Ignore CR in CR/NL combination
				continue;
			case '\n':
				if (brackets <= 0) {
					goto done1;
				}
			case '\t':
				c = ' ';
			case ' ':
				break;
		}	}
		P[lth++] = c;
	}
done1:
	ret = 1;
	P[lth] = 0;
	return(ret);
}

static int
getinputexpected(FILE *F, char **input_bufPP)
{
	char	*P;
	int P_size;
	int 	c, c1;
	long	start_line, lth;

	if (!F) {
		fprintf(stderr, "Input source for expected solution is null\n");
		exit(1);
	}
	if (!(P = *input_bufPP)) {
		P = malloc(4096);
		P_size = 4096;
		if (!P) {
			outofmemory();
		}
		*input_bufPP = P;
	}
	lth = start_line = 0;
	for (;;) {
		c = getc(F);
		if (feof(F)) {
			if (ferror(F)) {
				continue;
			}
			c = '\n';
		}
		if (lth >= (int) P_size) {
			P = realloc(P, lth+lth);
			P_size = lth+lth;
			if (!P) {
				outofmemory();
			}
			*input_bufPP = P;
		}
		if (c == '\r') {
			continue;	// Ignore CR
		}
		if (c == '\n') {
			for (; lth && ((c1 = P[lth-1]) == ' ' || c1 == '\t'); --lth);
			if ((lth - start_line) >= 4 && !strncmp(P+start_line, "</=>", 4)) {
				lth = start_line;
				break;
			}
			if (feof(F)) {
				return(0);
			}
			start_line = lth+1;
		}
		P[lth++] = c;
	}	
	P[lth] = 0;
	return(1);
}

static void
report_elapsed(timeT *afterP, timeT *beforeP)
{
	int hh, mm, sec, mill;

	mill  = ((int) (afterP->time-beforeP->time))*1000 + (afterP->millitm-beforeP->millitm);
	sec   = mill / 1000;
	mill -= (sec * 1000);
	mm	  = sec  / 60;
	sec  -= (mm  * 60);
	hh	  = mm	 / 60;
	mm	 -= (hh  * 60);
	printf("Elapsed: %d:%02d:%02d.%03d\n", hh, mm, sec, mill);
}

static void
finalize(void)
{
	finalizePD();
	finalizePatterns();
	finalizeVars();
	finalize_tmpPatterns();
}

static void
testit(testE test_case, Lptr lispP)
{
	extern Lptr CGUData;

	timeT	before, after;

	if (!state) {
		return;
	}
	if (eflag) {
		printf("</%s>\n", test_name[test_case]);
	}
	actual.m_lth = 0;

	if (state == -1) {
		print(&actual, "Execution prevented by input errors\n");
	} else {
		switch (test_case) {
		case test_parse:
		{
			printS(&actual, lispP);
			fputs(actual.m_P, stdout);
			fputc('\n', stdout);
			dump_lsedit(&lsedit, lispP);
			break;
		}
		case test_eval:
		{
			Lptr resultP;
			
			if (elapse_flag) {
				_ftime(&before);
			}
			resultP = assign(eval(lispP));
			printS(&actual, resultP);
			fputs(actual.m_P, stdout);
			fputc('\n', stdout);
			dump_lsedit(&lsedit, resultP);
			release(resultP);
			if (elapse_flag) {
				_ftime(&after);
				report_elapsed(&after, &before);
			}
			break;
		}
		case test_expand2:
		case test_expand1:
			if (state & 1) {
				if (state == 1) {
					print(&actual, "Script contained no PD definition\n");
				} else {
					print(&actual, "Script contained no RHS for additional PD\n");
				}
				break;
			} 
			if (elapse_flag) {
				_ftime(&before);
			}
			printEveryThing(&actual);
			if (!expand1()) {
				print(&actual, "\nexpand1 *failed*\n");
			} else {
				if (test_case == test_expand2) {
					if (!expand2()) {
						print(&actual, "\nexpand2 *failed*\n");
				}	}
			}
			printEveryThing(&actual);
			fputs(actual.m_P, stdout);
			if (elapse_flag) {
				_ftime(&after);
				report_elapsed(&after, &before);
			}
			dump_lsedit(&lsedit, 0);

			finalize();
			break;
		}
	}
	state	  = 0;
	test_case = default_test_case;
}

static int
regression_test(char *actualP, char *solutionP)
{
	char	*P1, *P2, *P3, *P4, *startP1, *startP2;
	int 	c1, c2;

	P1		= startP1 = actualP;

	// Remove all carriage returns (stupidity of NT \r\n convention)
	for (P3 = P4 = P1; ; ++P3) {
		if (*P3 != '\r') {
			*P4++ = *P3;
		}
		if (!*P3) {
			break;
	}	}

	P2 = startP2 = solutionP;
ok1:
	c1 = *P1++;
	switch (c1) {
	case '\n':
		startP1 = P1;
	case ' ':
	case '\t':
	case '\r':
		goto ok1;
	}
ok2:
	c2 = *P2++;
	switch (c2) {
	case '\n':
		startP2 = P2;
	case ' ':
	case '\t':
	case '\r':
		goto ok2;
	}

	if (c1 || c2) {
		if (c1 == c2) {
			goto ok1;
		}
		printf("Unexpected result:\n");
//		printf("<saw>\n%s</saw>\n\n", actualP);
		printf("<expected>\n%s</expected>\n\n", solutionP);
		printf("<change>\n");
		for (--P1; startP1 < P1; ++startP1) {
			putchar(*startP1);
		}
		fputs("*=>*", stdout);
		for (; (c1 = *P1) && c1 != '\r' && c1 != '\n'; ++P1) {
			putchar(c1);
		}
		fputs("\n----------\n", stdout);
		for (--P2; startP2 < P2; ++startP2) {
			putchar(*startP2);
		}
		fputs("*=>*", stdout);
		for (; (c2 = *P2) && c2 != '\r' && c2 != '\n'; ++P2) {
			putchar(c2);
		}
		putchar('\n');
		return(0);
	}
	return(1);
}

/*	
   The mainline now accepts the following options: 
   -A  abort the mainline when an error is detected.  This is a convenient
	   feature to use when one wants to easily see where errors occurred
	   in massive regression tests, which would otherwise leave the errors
	   buried somewhere in massive amounts of output.
  -c   Print a copyright notice and indicate when the executable was compiled.
  -e   Print the execution elapse time associated with each test performed.
  -E   Echo the stdin read to the stdout.
  -t <dir>
	   Use this specified dir as the root for relative script file names.
	   This permits the scripts to be kept somewhere central, etc.

  <scripts>*  Named script files to be run can appear as the final parameters
  of the command line.	If no named script file is specified the presumption
  is that the user will directly enter input via standard input.  In this
  case use '^Z' under NET to indicate EOF.

  Format of the input:

  Sample script files corresponding (roughly) to the 20+ unit_tests can be
  found under scripts.

  The command reader recognises as special lines of input:

  !<unix command> as an escape to shell
  @<script name>  as a request to run the specified script. Such requests will
				  stack and unstack correctly.	Eg: see scripts/all.
  <=> as the start of expected output marker
  </=> as the terminator of expected output marker.

  None of the above special lines of input are required, except for </=> which
  is required to terminate any preceeding <=>.
*/

int main(int argc, char *argv[]){
	
	int 	abort_on_error = 0; 		// Set to 1 if any error should cause immediate termination of the testbed


	int 	optind;
	char	*P;
	char	*filenameP = 0;
	char	*solutionP = 0;
	int 	ret 	   = 0;
	int 	c1, ret1;
	char	*inputP    = 0;
	int		input_size = 0;
	Lptr	tmp, desc, tmp1, input;

	for (optind = 1; optind < argc; ++optind) {
		P = argv[optind];
		if (*P != '-') {
			break;
		}
		switch (P[1]) {
		case 'A':
			abort_on_error = 1;
			break;
		case 'c':
			fprintf(stderr, "Copyright .. all rights strictly reserved\n");
			fprintf(stderr, "This executable created on " __DATE__ " at " __TIME__ );
			continue;
		case 'e':
			elapse_flag = 1;
			continue;
		case 'g':
			++optind;
			if (optind >= argc) {
				fprintf(stderr, "The -g option requires a location for lsedit\n");
				return(1);
			}
			lsedit.directoryP = argv[optind];
			continue;
		case 'h':
			lsedit.show_hash = 1;
			continue;
		case 'G':
			lsedit.show_CGUData = 1;
			continue;
		case 'V':
			lsedit.show_PopLog = 1;
			continue;
		case 'E':
			eflag = 1;
			continue;
		case 't':
			++optind;
			if (optind >= argc) {
				fprintf(stderr, "The -t option requires a test directory\n");
				return(1);
			}
			for (testdirP = argv[optind]; (c1 = *testdirP) == ' ' || c1 == '\t'; ++testdirP);
			if (!*testdirP) {
				fprintf(stderr, "Invalid test directory name\n");
				return(1);
			}
			continue;
		case 'T':
		{
			testE	test;

			++optind;
			if (optind >= argc) {
				fprintf(stderr, "The -T option requires test name\n");
				return(1);
			}
			test = lookup_test(argv[optind], 0);
			if (test == test_invalid) {
				fprintf(stderr, "Invalid test name '%s'\n", argv[optind]);
				return(1);
			}
			default_test_case = test;
			test_case		  = test;
			continue;
		}
		default:
			printf("Option '%s' is invalid\n\n", P);
			printf("Options:  -A/bort on error\n");
			printf("          -e/lapse time\n");
			printf("          -E/cho input\n");
			printf("          -c/opyright\n");
			printf("          -g/raph <dir containing lsedit>\n");
			printf("             (only relevant if -g/raph)\n");
			printf("             -h/ash show hash table\n");
			printf("             -G/CU  show GCUData\n");
			printf("             -V/er  show PopLog\n");
			printf("          -t/est directory for scripts\n");
			printf("          -T/est case\n");
			printf("Arguments: script filenames\n");
			return(1);
	}	}

	//********** Unit test cases ****************
//	unit_cases();

	if (optind < argc) {
		eflag = 1;
	} else {
		// Make stdin the input source
		pushinput(0);
	}

	initmemmgr();
	inittlisp();
	initpattern();
	initreasoner();
	initoptimizer();
	init_tmpPatterns();
	initVars();
	initcodegen();
	initsprint();
	sprint_init(&actual);

	do {
		if (optind < argc) {
			if (!pushinput(argv[optind++])) {
				if (abort_on_error) {
					goto abort;
				}
				continue;
		}	}
		
		for (;;) {
	
			if (!getinputcommand(stackP->F, &input_size, &inputP)) {
				testit(test_case, 0);
				if (!popinput()) {
					break;
				}
				continue;
			}
		
			switch (inputP[0]) {
				case '<': 
				{
					testE	test;
					char	*P1;
					
					P1 = inputP + 1;
					if (*P1 == '/') {
						++P1;
						test = lookup_test(P1, '>');
						if (test == test_invalid) {
							fprintf(stderr, "Unknown end tag %s\n", inputP);
							if (abort_on_error) {
								goto abort;
							}
							continue;
						}
						if (test != test_case) {
							fprintf(stderr, "Wrong test case for end tag %s\n", inputP);
							if (abort_on_error) {
								goto abort;
							}
							continue;
						}
						testit(test_case, 0);
						continue;
					} 

					test = lookup_test(P1, '>');
					if (test == test_invalid) {
						fprintf(stderr, "Unknown tag %s\n", inputP);
						if (abort_on_error) {
							goto abort;
						}
						continue;
					}
					if (test != test_regression) {
						test_case = test;
						continue;
					}

					if (!getinputexpected(stackP->F, &solutionP)) {
						fprintf(stderr, "Solution not terminated with </=>\n");
						if (abort_on_error) {
							goto abort;
						}
						continue;
					}

					testit(test_case,0);
					ret1 = 0;
					if (!actual.m_lth) {
						fprintf(stderr, "Solution given but no test result computed\n");
						ret1 = 2;
					} else if (!regression_test(actual.m_P, solutionP)) {
						ret1 = 2;
					}
						
					if (ret1) { 
						ret = ret1;
						if (stackP->filenameP) {
							printf("\nError in file %s\n\n", stackP->filenameP);
						}
						if (abort_on_error) {
							goto abort;
					}	}
					continue;
				}
				case '!': {
					if (!inputP[1]) {
						system("sh");
					} else {
						system(inputP+1);
					}
					continue;
				}
				case '@': {
					if (!pushinput(inputP+1)) {
						if (abort_on_error) {
							goto abort;
					}	}
					continue;
			}	}

			
			if (eflag) {
				if (!state) {
					printf("<%s>\n", test_name[test_case]);
				} else if (!(state & 1)) {
					printf("\n");
				}
				printf("%s\n", inputP);
			}
			
			if (state == -1) {
				continue;
			}
			switch (test_case) {
			case test_parse:
				input = lread(inputP);
				if (input == NULL) {
					state = -1;
				} else { 
					assign(input);
					state = 1;
				}
				testit(test_case, input);
				if (input != NULL) {
					release(input);
				}
				continue;
			case test_eval:
				input = lread(inputP);
				if (input == NULL) {
					state = -1;
				} else { 
					assign(input);

					gensym_init();
					tmp = assign(lread("PD1"));
					createPD(tmp);
					setPD(tmp);
					release(tmp);
			
					state = 1;
				}
				testit(test_case, input);
				if (input != NULL) {
					release(input);
				}
				continue;
			case test_expand1:
			case test_expand2:
				switch (++state) {
				case 0:
					state = -1;
					continue;
				case 1:
					gensym_init();
					tmp = assign(lread("PD1"));
					createPD(tmp);
					setPD(tmp);
					release(tmp);
					tmp = lread(inputP);
					if (tmp == NULL) {
						state = -1;
					} else {
						assign(tmp);
					}
					continue;
				case 2:
					desc = lread(inputP);
					if (desc == NULL) {
						state = -1;
					} else {
						assign(desc);
						addNode(tmp);
						pushStack(cdr(cdr(getCurrPD())),desc,tmp);
						release(desc);
					}
					release(tmp);  
					continue;
				default:
					if (state & 1) {
						tmp1 = lread(inputP);	// LHS
						if (tmp1 == NULL) {
							state = -1;
						} else {
							assign(tmp1);
						}
						continue;
					}
					tmp = lread(inputP);
					if (tmp == NULL) {
						state = -1;
					} else {
						assign(tmp);
						addGlobalIncDep(tmp1,tmp);
						release(tmp);
					}
					release(tmp1);
					continue;
				}
			} /* End case */
		}
	} while (optind < argc);

finish:
	while (stackP >= stack) {
		popinput();
	}
	sprint_free(&actual);
	free(inputP);
	inputP = 0;
	free(solutionP);
	solutionP = 0;
	finalize();
	finalizeHash();
	finish();

	fprintf(stderr, "<Continue>\n");
	getchar();
	return(ret);
abort:
	ret = 2;
	goto finish;
}
