/***************************************************************************
								  lsedit.c	 
							 -------------------
	Description 		   Driving logic to display stuff to lsedit
 ***************************************************************************/
/***************************************************************************
 *																		   *
 *	 This program is free software; you can redistribute it and/or modify  *
 *	 it under the terms of the GNU General Public License as published by  *
 *	 the Free Software Foundation; either version 2 of the License, or	   *
 *	 (at your option) any later version.								   *
 *																		   *
 ***************************************************************************/

#include "config.h"

#include <assert.h>
#include <sys/types.h>
#include <sys/timeb.h>
#include <time.h>
#include <stdio.h>
#include <stdlib.h>

#include "memmgr.h"
#include "tlisp.h"
#include "lsedit.h"

#define RESULT	((Lptr) 0)
#define CGUDATA ((Lptr) 1)
#define POPLOG	((Lptr) 2)
#define HASH	((Lptr) 3)

/* This routine generates unique id's for atomic values by using the arc that addresses them
 * as a uniquely identifying characteristic of such values.  This creates the desired logical
 * separation of atomic nodes sharing the same value, from their underlying shared nature.
 *
 * ptype is a character:
 *	 A -- Addressed by parentP's CAR
 *	 D -- Addressed by parentP's CDR
 *	 V -- Addressed by parentP's DEF
 *	 H -- Addressed by hash table pointer
 *	 N -- A node which is not some sort of special case such as cons
 *	 T -- Top level
 */

static void
dump_lsedit_id(const lseditT *lseditP, void *parentP, int ptype, Lptr lispP, FILE *F)
{
	if (ptype == 'H') {
		fprintf(F, "H%p", lispP);
		return;
	}
	switch (Vtype(lispP)) {
	case INTTYPE:
	case STRTYPE:
	case NILTYPE:
	case UNDEFTYPE:
	case ERRORTYPE:
		fprintf(F, "%c%px%p", ptype, parentP, lispP);
		return;
	}
	fprintf(F, "N%p", lispP);
}

static void
dump_lsedit_type(Lptr lispP, FILE *F)
{
	switch (Vtype(lispP)) {
	case CONSTYPE:
		if (getVersion(lispP) == 0) {
			fprintf(F, " cons\n");
		} else {
			fprintf(F, " vcons\n");
		}
		break;
	case INTTYPE:
		fprintf(F, " int\n");
		break;
	case STRTYPE:
		fprintf(F, " string\n");
		break;
	case NILTYPE:
		fprintf(F, " nil\n");
		break;
	case UNDEFTYPE:
		fprintf(F, " undef\n");
		break;
	case ERRORTYPE:
		fprintf(F, " error\n");
		break;
	case FRAGTYPE:
		fprintf(F, " frag\n");
		break;
	case FREETYPE:
		fprintf(F, " free\n");
		break;
	default:
		fprintf(F, " unknown\n");
		break;
	}
}

static void
dump_lsedit_entities(const lseditT *lseditP, const char *containerP, Lptr parentP, int arc, Lptr lispP, FILE *F)
{
	celltypeE	type;
	Lptr		lisp1P;
	int 		marked;

	type   = Vtype(lispP);
	marked = Vismarked(lispP);
	if (marked) {
		switch (type) {
		case INTTYPE:
		case STRTYPE:
		case NILTYPE:
		case UNDEFTYPE:
		case ERRORTYPE:
			// Even if we are marked we are called from something not marked
			// So we have to create an entry for ourselves
			break;
		case CONSTYPE:
			return;
	}	}
	
	fprintf(F, "$INSTANCE ");
	dump_lsedit_id(lseditP, parentP, arc, lispP, F);
	dump_lsedit_type(lispP, F);
	if (containerP) {
		fprintf(F, "contain %s ", containerP);
		dump_lsedit_id(lseditP, parentP, arc, lispP, F);
		fputc('\n', F);
	}
	switch (type) {
	case CONSTYPE:
		Vsetmark(lispP);
		lisp1P		 = Car(lispP);
		dump_lsedit_entities(lseditP, containerP, lispP, 'A', lisp1P, F);
		lisp1P		 = Cdr(lispP);
		dump_lsedit_entities(lseditP, containerP, lispP, 'D', lisp1P, F);
		break;
	case INTTYPE:
	case STRTYPE:
		if (!marked) {
			lisp1P = Def(lispP);

			if (lisp1P) {
				if (Vtype(lisp1P) != UNDEFTYPE) {
					Vsetmark(lispP);
					dump_lsedit_entities(lseditP, containerP, lispP, 'V', lisp1P, F);
		}	}	}
		break;
	}
}	

static void
dump_lsedit_relations(const lseditT *lseditP, Lptr parentP, int arc, Lptr lispP, FILE *F)
{
	celltypeE	type;
	Lptr		lisp1P;
	int 		marked;

	type   = Vtype(lispP);
	marked = Vismarked(lispP);
	if (marked) {
		switch (type) {
		case INTTYPE:
		case STRTYPE:
		case NILTYPE:
		case UNDEFTYPE:
		case ERRORTYPE:
			// Even if we are marked we are called from something not marked
			// So we have to create a relation for our def pointer with our newly created alias
			break;
		case CONSTYPE:
			return;
	}	}

	switch (type) {
	case CONSTYPE:
		lisp1P = Car(lispP);
		fprintf(F, "car ");
		dump_lsedit_id(lseditP, 0, 0, lispP, F);
		fputc(' ', F);
		dump_lsedit_id(lseditP, lispP, 'A', lisp1P, F);
		fputc('\n', F);
		Vsetmark(lispP);
		dump_lsedit_relations(lseditP, lispP, 'A', lisp1P, F);
		lisp1P = Cdr(lispP);
		fprintf(F, "cdr ");
		dump_lsedit_id(lseditP, 0, 0, lispP, F);
		fputc(' ', F);
		dump_lsedit_id(lseditP, lispP, 'D', lisp1P, F);
		fputc('\n', F);
		dump_lsedit_relations(lseditP, lispP, 'D', lisp1P, F);
		break;
	case INTTYPE:
	case STRTYPE:
		if (lseditP->show_hash && parentP != HASH) {
			if (Nexthash(lispP)) {
				fprintf(F, "hash ");
				dump_lsedit_id(lseditP, parentP, arc, lispP, F);
				fputc(' ', F);
				dump_lsedit_id(lseditP, HASH, 'T', lispP, F);
				fputc('\n', F);
		}	}
		lisp1P = Def(lispP);
		if (lisp1P) {
			if (Vtype(lisp1P) != UNDEFTYPE) {
				fprintf(F, "def ");
				dump_lsedit_id(lseditP, parentP, arc, lispP, F);
				fputc(' ', F);
				dump_lsedit_id(lseditP, lispP, 'V', lisp1P, F);
				fputc('\n', F);
		}	}
		if (!marked) {
			Vsetmark(lispP);
			if (lisp1P) {
				if (Vtype(lisp1P) != UNDEFTYPE) {
					dump_lsedit_relations(lseditP, lispP, 'V', lisp1P, F);
		}	}	}
		break;
}	}

static void
dump_lsedit_attribute(Lptr lispP, FILE *F)
{
	switch (Vtype(lispP)) {
	case STRTYPE:
	{
		Lptr		frag;
		int 		c;
		const char	*source;
		const char	*max;
		
		fprintf(F, " { title=%p refcnt=%d description=\"Refcnt %d\" label=\"", lispP, Rcount(lispP), Rcount(lispP));
		for (frag = Start_frag(lispP); frag != NULL ; frag = Next_frag(frag)) {
			source = Strdata(frag);
			max    = source + sizeof(Strdata(frag));
			for (; source < max && (c = *source++); fputc(c, F));
		}
		fprintf(F, "\" }\n");
		return;
	}
	case INTTYPE:
		fprintf(F, " { title=%p refcnt=%d description=\"Refcnt %d\" label=\"%d\" }\n", lispP, Rcount(lispP), Rcount(lispP), Intval(lispP));
		return;
	case CONSTYPE:
		if (getVersion(lispP)) {
			fprintf(F, " { title=%p refcnt=%d label=\"Version %d\" description=\"Refcnt %d\" }\n", lispP, Rcount(lispP), Version(lispP), Rcount(lispP));
		} else {
			fprintf(F, " { title=%p refcnt=%d label=cons description=\"Refcnt %d\" }\n", lispP, Rcount(lispP), Rcount(lispP));
		}
		return;
	case UNDEFTYPE:
		fprintf(F, " { title=%p label=\"*undef*\" }\n", lispP);
		return;
	case ERRORTYPE:
		fprintf(F, " { title=%p label=\"*error*\" }\n", lispP);
		return;
	case NILTYPE:
		fprintf(F, " { title=%p label=nil }\n", lispP);
		return;
	case FRAGTYPE:
		fprintf(F, " { title=%p label=frag }\n", lispP);
		return;
	case FREETYPE:
		fprintf(F, " { title=%p label=free }\n", lispP);
		return;
	default:
		assert(0);
		return;
	}
}	

static void
dump_lsedit_entity_attributes(const lseditT *lseditP, Lptr parentP, int arc, Lptr lispP, FILE *F)
{
	Lptr		lisp1P;
	celltypeE	type;
	int 		marked;

	type   = Vtype(lispP);
	marked = Vismarked(lispP);
	if (marked) {
		switch (type) {
		case INTTYPE:
		case STRTYPE:
		case NILTYPE:
		case UNDEFTYPE:
		case ERRORTYPE:
			// Even if we are marked we are called from something not marked
			// So we have to create a relation for our def pointer with our newly created alias
			break;
		case CONSTYPE:
			return;
	}	}

	dump_lsedit_id(lseditP, parentP, arc, lispP, F);
	dump_lsedit_attribute(lispP, F);

	switch (type) {
	case CONSTYPE:
		Vsetmark(lispP);
		dump_lsedit_entity_attributes(lseditP, lispP, 'A', Car(lispP), F);
		dump_lsedit_entity_attributes(lseditP, lispP, 'D', Cdr(lispP), F);
		return; 
	case STRTYPE:
	case INTTYPE:
		lisp1P = Def(lispP);
		if (!marked) {
			if (lisp1P) {
				if (Vtype(lisp1P) != UNDEFTYPE) {
					Vsetmark(lispP);
					dump_lsedit_entity_attributes(lseditP, lispP, 'V', lisp1P, F);
		}	}	}
		return;
	}
}	

static int
dump_hash_entries(const lseditT *lseditP, const char *containerP, FILE *F)
{
	extern Lptr hashtable[MAXHASHSIZE];

	Lptr		*headPP, *endPP, listP;
	int 		seen = 0;
	static int	cnt;

	seen = 0;
	headPP = hashtable;
	for (endPP	= hashtable + MAXHASHSIZE; headPP < endPP; ++headPP) {
		listP = *headPP;
		if (!vNull(listP)) {
			seen = 1;
			for (; !vNull(listP); listP = Nexthash(listP)) {
				dump_lsedit_entities(lseditP, containerP, HASH, 'T', listP, F);
	}	}	}
	return(seen);
}

static void
dump_hash_relations(const lseditT *lseditP, FILE *F)
{
	extern Lptr hashtable[MAXHASHSIZE];

	Lptr		*headPP, *endPP, listP, prev_cdrP, *prev_carPP;
	static int	cnt;

	headPP	   = hashtable;
	prev_carPP = 0;
	for (endPP	= hashtable + MAXHASHSIZE; headPP < endPP; ++headPP) {
		listP = *headPP;
		if (!vNull(listP)) {
			if (prev_carPP) {
				fputs("cdr ", F);	// Lay the top level of the hash table out horizontally
				dump_lsedit_id(lseditP, HASH, 'T', *prev_carPP, F);
				fputc(' ', F);
				dump_lsedit_id(lseditP, HASH, 'T', listP, F);
				fputc('\n', F);
			}
			prev_carPP = headPP;
		
			for (prev_cdrP = listP; ; prev_cdrP = listP) {
				dump_lsedit_relations(lseditP, HASH, 'T', listP, F);
				listP = Nexthash(listP);
				if (vNull(listP)) {
					break;
				}
				fputs("car ", F);	// Lay each chain out vertically
				dump_lsedit_id(lseditP, HASH, 'T', prev_cdrP, F);
				fputc(' ', F);
				dump_lsedit_id(lseditP, HASH, 'T', listP, F);
				fputc('\n', F);
}	}	}	}

static void
dump_hash_attributes(const lseditT *lseditP, FILE *F)
{
	extern Lptr hashtable[MAXHASHSIZE];

	Lptr		*headPP, *endPP, listP;
	int 		seen = 0;
	static int	cnt;

	headPP = hashtable;
	for (endPP	= hashtable + MAXHASHSIZE; headPP < endPP; ++headPP) {
		for (listP = *headPP; !vNull(listP); listP = Nexthash(listP)) {
			dump_lsedit_entity_attributes(lseditP, HASH, 'T', listP, F);
}	}	}

static void
unmark(Lptr lispP)
{
	if (Vismarked(lispP)) {
		Vclrmark(lispP);

		switch (Vtype(lispP)) {
		case INTTYPE:
		case STRTYPE:
			unmark(Def(lispP));
			break;
		case CONSTYPE:
			unmark(Car(lispP));
			unmark(Cdr(lispP));
			break;
	}	}
}	

static void
unmark_hash(void)
{
	extern Lptr hashtable[MAXHASHSIZE];

	Lptr		*headPP, *endPP, listP;
	int 		seen = 0;
	static int	cnt;

	headPP = hashtable;
	for (endPP	= hashtable + MAXHASHSIZE; headPP < endPP; ++headPP) {
		for (listP = *headPP; !vNull(listP); listP = Nexthash(listP)) {
			unmark(listP);
}	}	}

void
dump_lsedit(const lseditT *lseditP, Lptr resultP)
{
	extern	Lptr CGUData;
	extern	Lptr PopLog;

	static const char *introP = 
		"SCHEME TUPLE :\n\n"
		"$INHERIT structure $ENTITY\n"
		"$INHERIT node $ENTITY\n"
		"$INHERIT cons node\n"
		"$INHERIT vcons cons\n"
		"$INHERIT atom node\n"
		"$INHERIT string atom\n"
		"$INHERIT int atom\n"
		"$INHERIT nil atom\n"
		"$INHERIT undef atom\n"
		"$INHERIT error node\n"
		"$INHERIT frag error\n"
		"$INHERIT free error\n"
		"$INHERIT unknown error\n\n"
		"contain structure node\n"
		"car cons node\n"
		"cdr cons node\n"
		"def atom node\n"
		"hash atom atom\n"
		"SCHEME ATTRIBUTE :\n\n"
		"$ENTITY { color = (255 255 255) labelcolor = (0 51 51) class_style = 1 }\n"
		"structure { color = (51 255 51)   labelcolor = (0 0 0)      class_style = 1  class_image = 8  }\n"
		"node    { color = (255 204 204) labelcolor = (0 0 0)   class_style = 1 }\n"
		"cons    { color = (255 102 102) labelcolor = (0 0 0)   class_style = 0 }\n"
		"vcons   { color = (255 153 153) labelcolor = (0 0 0)   class_style = 0 }\n"
		"atom    { color = (255 204 204) labelcolor = (0 0 0)   class_style = 1 }\n"
		"int     { color = (51 255 51)   labelcolor = (128 128 128) class_style = 5 }\n"
		"string  { color = (255 255 0)   class_style = 6 }\n"
		"nil     { color = (51 255 255)  labelcolor = (0 0 0)      class_style = 9 }\n"
		"undef   { color = (255 0 255)   labelcolor = (0 0 0)      class_style = 10 }\n"
		"free    { color = (0 0 255)     labelcolor = (255 255 0)  class_style = 16 class_image = 16 }\n"
		"error   { color = (255 204 204) labelcolor = (0 0 0)      class_style = 16 class_image = 16 }\n"
		"unknown { color = (0 51 0)      labelcolor = (255 51 255) class_style = 16 class_image = 16 }\n"
		"frag    { color = (0 0 255)     labelcolor = (255 255 0)  class_style = 16 class_image = 16 }\n\n"

		"(car)   { color = (0 0 0)   class_style = 0 }\n"
		"(cdr)   { color = (255 0 0) class_style = 0 }\n"
		"(def)   { color = (0 0 255) class_style = 0 }\n"
		"(hash)  { color = (51 255 51) class_style = 0 }\n\n"
		"FACT TUPLE :\n\n";

	FILE	*F;
	Lptr	CGUData1, PopLog1;
	int 	hash = 0;
	char	*containerP = 0;
	int 	use_container = 0;
	const	char *directoryP;
	const	char *targetP;

	if (!(directoryP = lseditP->directoryP)) {
		return;
	}
	CGUData1 = 0;
	PopLog1  = 0;

	if (lseditP->show_hash) {
		targetP = "-N Hashtable";
		++use_container;
	} 
	if (lseditP->show_PopLog) {
		targetP = "-N PopLog";
		++use_container;
		PopLog1  = PopLog;
	}
	if (lseditP->show_CGUData) {
		targetP = "-N CGUData";
		++use_container;
		CGUData1 = CGUData;
	}
	
	if (resultP) {
		targetP = "-N Result";
		++use_container;
	}

	switch (use_container) {
	case 0:
		return;
	case 1:
		targetP = "";
		use_container = 0;
	default:
		use_container = 1;
	}

	F = fopen("gtab.ta", "w");
	if (!F) {
		fprintf(stderr, "Can't open gtab.ta\n");
	} else {
		char	command[1024];

		fputs(introP, F);
			
		if (resultP) {
			containerP = 0;
			fputs("\n// Result entities\n\n", F);
            if (use_container) {
                fputs("$INSTANCE Result structure\n", F);
                containerP = "Result";
            }
            dump_lsedit_entities(lseditP, containerP, RESULT, 'T', resultP, F);
        }
        if (CGUData1) {
            containerP = 0;
            fputs("\n// CGUData entities\n\n", F);
			if (use_container) {
				fputs("$INSTANCE CGUData structure\n", F);
				containerP = "CGUData";
			}
			dump_lsedit_entities(lseditP, containerP, CGUDATA, 'T', CGUData1, F);
		}
		if (PopLog1) {
			containerP = 0;
			fputs("\n// PopLog entities\n\n", F);
            if (use_container) {
                fputs("$INSTANCE PopLog structure\n", F);
                containerP = "PopLog";
            }
            dump_lsedit_entities(lseditP, containerP, POPLOG, 'T', PopLog1, F);
        }
        if (lseditP->show_hash) {
            containerP = 0;
            fputs("\n// Hash entities\n\n", F);
			if (use_container) {
				fputs("$INSTANCE Hashtable structure\n", F);
				containerP = "Hashtable";
			}
			hash = dump_hash_entries(lseditP, containerP, F);
			unmark_hash();
		}

		if (PopLog1) {
			unmark(PopLog1);
		}

		if (CGUData1) {
			unmark(CGUData1);
		}
		if (resultP) {
			unmark(resultP);
		}
	
		// Now dump relations

		if (resultP) {
			fputs("\n// Result relations\n\n", F);
            dump_lsedit_relations(lseditP, RESULT, 'T', resultP, F);
        }
        if (CGUData1) {
            fputs("\n// CGUData relations\n\n", F);
			dump_lsedit_relations(lseditP, CGUDATA, 'T', CGUData1, F);
		}
		if (PopLog1) {
			fputs("\n// PopLog relations\n\n", F);
            dump_lsedit_relations(lseditP, POPLOG, 'T', PopLog1, F);
        }
        if (hash) {
            fputs("\n// Hash relations\n\n", F);
			dump_hash_relations(lseditP, F);
			unmark_hash();
		}

		if (PopLog1) {
			unmark(PopLog1);
		}

		if (CGUData1) {
			unmark(CGUData1);
		}
		if (resultP) {
			unmark(resultP);
		}

		fprintf(F, "FACT ATTRIBUTE :\n\n");
		if (resultP) {
			fputs("\n// Result attributes\n\n", F);
            dump_lsedit_entity_attributes(lseditP, RESULT, 'T', resultP, F);
        }
        if (CGUData1) {
            fputs("\n// CGUData attributes\n\n", F);
			dump_lsedit_entity_attributes(lseditP, CGUDATA,'T', CGUData1, F);
		}
		if (PopLog1) {
			fputs("\n// PopLog attributes\n\n", F);
            dump_lsedit_entity_attributes(lseditP, POPLOG, 'T', PopLog1, F);
        }
        if (hash) {
            fputs("\n// Hash attributes\n\n", F);
			dump_hash_attributes(lseditP, F);
			unmark_hash();
		}
		if (PopLog1) {
			unmark(PopLog1);
		}

		if (CGUData1) {
			unmark(CGUData1);
		}
		if (resultP) {
			unmark(resultP);
		}
	
		fclose(F);
		sprintf(command, "java.exe -classpath %s;%s/bunch.jar lsedit.LandscapeEditorFrame  -l lisp %s gtab.ta", directoryP, directoryP, targetP);
		fputs(command, stderr);
		fputc('\n', stderr);
		system(command);
}	}

