/***************************************************************************
								  cgu.c 
							 -------------------
	Description 		  C file for Common Graph Utility (CGU)
 ***************************************************************************/
/***************************************************************************
 *																		   *
 *	 This program is free software; you can redistribute it and/or modify  *
 *	 it under the terms of the GNU General Public License as published by  *
 *	 the Free Software Foundation; either version 2 of the License, or	   *
 *	 (at your option) any later version.								   *
 *																		   *
 ***************************************************************************/

#include "config.h"
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "memmgr.h"
#include "tlisp.h"
#include "err.h"
#include "pattern.h"
#include "sprint.h"
#include "cgu.h"

Lptr CGUData = NULL ;
/* (<pointer to the list of global LHSs> <list of PDs>)
   In other words, it is a list of cons cells. The "car" of first cell point to 
   the list of LHSs and "car"s of all other cons cells point to PDs.*/

Lptr PDList = NULL;
/* It is a pointer to the head of list of PDs
   In current design, it is equal to cdr(CGUData) */

Lptr currPD=NULL ;
/*Pointer to current PD. To modify it use setPD function */

Lptr tempabcd = NULL; 
Lptr pats[33];

char * snrettap1 =
  "("														  //pats[number]
	" (waitf < f > n >* pf (atomic > C) > D)"				  //1
	" pf"													  //2
	" (atomic < C)" 										  //3
	" D"													  //4
	" n"													  //5
	" (not (atomic < C))"									  //6
	" (waitc (atomic < C) < n < D)" 						  //7
	" g"													  //8
	" pftail"												  //9
	" (waitf  < g < n << pftail (atomic < C) < D)"			  //10
	" n2"													  //11
	" (ineq < n2)"											  //12
	" n1"													  //13
	" (ineq < n1)"											  //14
	" (edge > n4 < f)"										  //15
	" (atomic QVAR)"										  //16
	" (waitc (atomic < C) > n3 > D)"						  //17
	" n3"													  //18
	" (edge > n3 < f)"										  //19
	" (redge > n3 > f)" 									  //20
	" (redge > n4 < f)" 									  //21
	" n4"													  //22
	" PDNDROOT" 											  //23
	" lbl"													  //24
	" (edge > n1 < lbl)"									  //25
	" (eqclass > N1)"										  //26
	" N1"													  //27
	" GlobalT"												  //28
	" and"													  //29
	" (copyQvar < node1)"									  //30
	" (atomic > C)" 										  //31
	" top"													  //32
	" or"													  //33
  ")" ;


void gtab_crash(int err_code) 
{
  printf("System crash with error code %d.\n", err_code);
  exit(err_code);
}

void init_tmpPatterns(void) {
  int i;
  Lptr iter ;
  setSptr(&tempabcd, lread(snrettap1));
  pats[0] = getNil();
  for (i = 1,iter=tempabcd; i<=33; iter=cdr(iter),i++)
	pats[i] = car(iter);
}

void finalize_tmpPatterns(void) {
  setSptr(&tempabcd, NULL);
}

Lptr createPD (Lptr PDid) {
  Lptr newPD ;
  if (CGUData == NULL)							  /* if CGUData is vNull */
	setSptr(&CGUData, cons(getNil(), getNil()));/* then initialize it */

  /* Initialize a new PD: PDid, PopList, Stack1(versioned), Stack2(versioned),
	 NodeList (versioned) */ 
  newPD = cons(PDid, cons(getNil(), cons(getNil(), cons(getNil(), cons(getNil(),getNil())))));

  /* Add the new PD into the CGUDate. It is added to the front the PD list */
  rplacd(CGUData, cons(newPD,cdr(CGUData)));
  
  setSptr(&PDList, cdr(CGUData));
  return newPD;
}

/* Set the current PD to the one input as PDid */
Lptr setPD(Lptr PDid) {
   Lptr temp;

   for (temp=PDList; !vNull(temp) && (car(car(temp)) != PDid) ; temp=cdr(temp));
   if (vNull(temp)) {
	  fprintf(stderr, "Error in setPD. Required id not found in PDList");
	  gtab_crash(ERR_PDID_NOT_MATCH);
   }
   else {
	  setSptr(&currPD, car(temp));
   }
   return currPD;
}


/* Return the pointer to the current PD */
Lptr getCurrPD(void) {
   return currPD;
}

void deleteLoops (Lptr pd) {
  Lptr iter;
  for (iter = car(cdr(cdr(cdr(cdr(pd))))) ; !vNull(iter) ; iter = cdr(iter)) {
	remprop(car(iter),car(currPD));
  }
}

void deletePD (Lptr PDid) {
  Lptr iter ;
  if (currPD != NULL && eq(car(currPD),PDid)) {
	printf("WARNING: Deleting the PD pointed to by currPD. Now currPD will be NULL \n");
	setSptr(&currPD, NULL);
  }
  for (iter = CGUData ; !vNull(cdr(iter)) ; iter=cdr(iter)) {
	if (eq(car(car(cdr(iter))),PDid)) {
	  deleteLoops(car(cdr(iter)));
	  iter = rplacd(iter,cdr(cdr(iter)));
	}
  }
  setSptr(&PDList, cdr(CGUData));
}


void finalizePD(void) {
  Lptr iter ;

  if (CGUData) {
	for (iter = car(CGUData) ; !vNull(iter) ; iter = cdr(iter)) {
		remprop(car(iter),pats[28]); // pats[28]="GlobalT"
	}
	for (iter = cdr(CGUData); !vNull(iter) ; iter=cdr(iter)) {
		deleteLoops(car(iter));
	}
	setSptr(&CGUData, NULL);
  }
  setSptr(&currPD, NULL);
  setSptr(&PDList, NULL);
  resetLogVer();
  finalizeNNFControl();
}

Lptr createNode(Lptr lbl) {
  char * nd = "ND" ;
  Lptr newNode ;
  if (!vNull(lbl)) {
	return (addNode(lbl));
  }
  else {
	newNode = gensym(nd);
	return (addNode(newNode));
  }
}

Lptr addNode(Lptr NodeId) {
  // "NodeId is an atom...
  Lptr descList;
 
  if (nullEval(NodeId) || vNull(getprophead(NodeId,car(currPD)))) {
	descList =	cons(getNil(),getNil());
	putprop(NodeId, car(currPD), descList);
	vaddList(cdr(cdr(cdr(cdr(currPD)))),NodeId);
  }
  if (!vNull(getprophead(pats[32],pats[28])))
	pushStack(cdr(cdr(currPD)),getprop(pats[32],pats[28]),NodeId);

  return NodeId;
}

Lptr addGlobalIncDep (Lptr LHS, Lptr RHS) {
  Lptr LHS1,RHS1,oldRHS;

  if (!vAtom(LHS)) {
	if (Match(pats[31],LHS)){								   //pats[31] = (atomic > C)
	  LHS1 = getBindVal(lread("C"));
	  RHS1 = RHS;
	}  else {
	  RHS1 = cons(pats[33],cons(RHS,cons(lread("not"),cons(LHS,getNil()))));   //pats[33] = or
	  LHS1 = pats[32];										   //pats[32] = (top)
	}
  }
  else {
	LHS1 = LHS; RHS1 = RHS;
  }

  if (CGUData == NULL)									 // if CGUData is vNull
	setSptr(&CGUData, cons(getNil(), getNil()));	   // then initialize it

  if (nullEval(LHS1) || vNull(getprophead(LHS1,pats[28]))) { //pats[28] = "GlobalT"
	putprop(LHS1, pats[28], RHS1);
	rplaca(CGUData, cons(LHS1, car(CGUData)));
  } else {
	oldRHS = getprop(LHS1,pats[28]);
	RHS1 = cons(pats[29],cons(oldRHS,cons(RHS1,getNil())));
	putprop(LHS1, pats[28], RHS1);								//pats[28] = "GlobalT"
  }
  return (LHS1);
}

Lptr addLocalIncDep (Lptr LHS, Lptr RHS) {

  return (addGlobalIncDep(LHS,RHS));
}


Lptr newQC(Lptr LHS, Lptr RHS) {
   Lptr QC;
   QC = cons(lread("and"), cons(LHS, cons(cons(lread("not"), cons(RHS, getNil())), getNil())));
   negationNormalForm(assign(QC));
   return(decrement(QC));
}

void add_Edge(Lptr node1,Lptr node2,Lptr label) {
  // Note: node1,node2 and label -- all 3 are atoms
  Lptr pf, newDesc;

  putBindVal(lread("lbl"),label);
  if (vNull(isDescInNode(pats[25],node1,Do_Not_Delete))) {	 //pats[25]="(edge > n1 < lbl)"

	newDesc = cons(lread("edge"),cons(node2,cons(label,getNil())));
	addDescription(node1,newDesc);
	newDesc = cons(lread("redge"),cons(node1,cons(label,getNil())));
	addDescription(node2,newDesc);

	while (!vNull(isDescInNode(pats[1],node1,Delete))){ 	 //pats[1]="(waitf < f > n >* pf (atomic > C) > D)"
	  // Note: "if" changed to "while"

	  pf = assign(cons(getBindVal(lread("f")),getBindVal(lread("pf"))));
	  checkPath(node1,getBindVal(lread("n")),pf,getBindVal(lread("C")),getBindVal(lread("D")));
	  release(pf);

//		pf = getBindVal(pats[2]);								//pats[2] = "pf"
//		if (vNull(pf)) {
//		  if(!vNull(isDescInNode(pats[3],node2,Do_Not_Delete))){ //pats[3] = "(atomic < C)"
//			D = getBindVal(pats[4]);							//pats[4] = "D"
//			n = getBindVal(pats[5]);							//pats[5] = "n"
//			pushStack(cdr(cdr(currPD)),D,n);
//		  }
//		  else {
//			if(vNull(isDescInNode(pats[6],node2,Do_Not_Delete))){//pats[6]="(not (atomic < C))"
//			  tmp = assign(Build(pats[7])); 					//pats[7] = "(waitc (atomic < C) < n < D)"
//			  addDescription(node2,tmp);
//			  release(tmp);
//			}
//		  }
//		}
//		else {
//		  g = car(pf);
//		  pftail = cdr(pf);
//		  putBindVal(pats[8],g);								//pats[8] = "g"
//		  putBindVal(pats[9],pftail);							//pats[9] = "pftail"
//		  tmp = assign(Build(pats[10]));						//pats[10]= "(waitf < g < n << pftail (atomic < C) < D)"
//		  addDescription(node2,tmp);
//		  release(tmp);
//		}
	}
  }
}

boolean exists_Ineqality_Edge(Lptr node1, Lptr node2) {
  putBindVal(pats[11],getEqClass(node2));								//pats[11] = "n2"
  if (!vNull(isDescInNode(pats[12],node1,Do_Not_Delete)))	 //pats[12] = " (ineq < n2)"
	return TRUE;
  putBindVal(pats[13],getEqClass(node1));								//pats[13] = "n1"
  if (!vNull(isDescInNode(pats[14],node2,Do_Not_Delete)))	 //pats[14] = "(ineq < n1)"
	return TRUE;
  return FALSE;
}

boolean setEqClass(Lptr n1, Lptr n2) {
  Lptr Dlist1,iter,iter1,tmp,D,n3,n4,pf,tmpL;
  Lptr eqlist = assign(cons(getNil(),getNil()));
  Dlist1 = car(getprophead(n1, car(currPD)));
  //Dlist2 = cdr(car(getprophead(n2,car(currPD))));
  tmpL = assign(lread("((atomic > C) (edge > n3 > f) (not (atomic > C)) (waitc (atomic > C) > n3 > D) (waitf > f > n >* pf (atomic > C) > D))"));

  for (iter = Dlist1; !vNull(cdr(iter)) ; ){
	iter1 = cdr(iter);
	if (!Match(pats[16],car(iter1))) {					   //pats[16] = "(atomic QVAR)"

	  if (vNull(isDescInNode(car(iter1),n2,Do_Not_Delete))) {

		if (Match(car(tmpL),car(iter1))) {				  //copying (atomic C)
		  if (!vNull(isDescInNode(pats[6],n2,Do_Not_Delete))) { //pats[6] = "(not (atomic < C))"
			release(tmpL); release(eqlist);
			return FALSE;
		  }
		  addDescription(n2,car(iter1));

		  if(!vNull(isDescInNode(pats[17],n2,Delete))) {	 //pats[17] = "(waitc (atomic < C) > n3 > D)"
			D = getBindVal(pats[4]);						//pats[4] = "D"
			n3 = getBindVal(pats[18]);						//pats[18] = "n3"
			pushStack(cdr(cdr(getCurrPD())),D,n3);
		  }
		}

		else if (Match(car(cdr(tmpL)),car(iter1))) {	 //copying (edge n3 f)
		  if (!vNull(isDescInNode(pats[15],n2,Do_Not_Delete))) { //pats[15]="(edge > n4 < f)"
			vaddList(eqlist,getBindVal(lread("n3")));
			vaddList(eqlist,getBindVal(lread("n4")));
		  }
		  else {
			while (!vNull(isDescInNode(pats[1],n2,Delete))) {	   //pats[1] = "(waitf < f > n >* pf (atomic > C) > D)"

			  pf = assign(cons(getBindVal(lread("f")),getBindVal(lread("pf"))));
			  checkPath(n2,getBindVal(lread("n")),pf,getBindVal(lread("C")),getBindVal(lread("D")));
			  release(pf);

//				pf = getBindVal(pats[2]);						//pats[2] = "pf"
//				n3 = getBindVal(pats[18]);						//pats[18] = "n3"
//				if (vNull(pf)) {
//
//				  if(!vNull(isDescInNode(pats[3],n3,Do_Not_Delete))){//pats[3] = "(atomic < C)"
//					n = getBindVal(pats[5]);					//pats[5] = "n"
//					D = getBindVal(pats[4]);					//pats[4] = "D"
//					pushStack(car(currPD),D,n);
//				  }
//				  else {
//
//					if(vNull(isDescInNode(pats[16],n3,Do_Not_Delete))){//pats[6] = "(not (atomic < C))"
//					  tmp = assign(Build(pats[7])); 			//pats[7] = "(waitc (atomic < C) < n < D)"
//					  addDescription(n3,tmp);
//					  release(tmp);
//					}
//				  }
//				}
//				else {
//				  putBindVal(pats[8],car(pf));				   //pats[8] = "g"
//				  putBindVal(pats[9],cdr(pf));				   //pats[9] = "pftail"
//				  tmp = assign(Build(pats[10]));				//pats[10] = "(waitf < g < n << pftail (atomic < C) < D)"
//				  addDescription(n3,tmp);
//				  release(tmp);
//				}
			}
			addDescription(n2,car(iter1));
		  }
		}

		else if (Match(car(cdr(cdr(tmpL))),car(iter1))) {	   // copying (not (atomic > C))
		  if (!vNull(isDescInNode(pats[3],n2,Do_Not_Delete))){	//pats[3] = "(atomic < C)"
			release(tmpL); release(eqlist);
			return FALSE;
		  }
		  addDescription(n2,car(iter1));
		  while(!vNull(isDescInNode(pats[17],n2,Delete))) ;

		}

		else if (Match(car(cdr(cdr(cdr(tmpL)))),car(iter1))) {	// copying (waitc (atomic > C) > n3 > D)

		  if(!vNull(isDescInNode(pats[3],n2,Do_Not_Delete))){//pats[3] = "(atomic < C)"
			D = getBindVal(pats[4]);						//pats[4] = "D"
			n3 = getBindVal(pats[18]);						//pats[18] = "n3"
			pushStack(cdr(cdr(getCurrPD())),D,n3);
		  }
		  else addDescription(n2,car(iter1));
		}

		else if (Match(car(cdr(cdr(cdr(cdr(tmpL))))),car(iter1))) {  // copying (waitf > f > n >* pf (atomic > C) > D)

		  if(!vNull(isDescInNode(pats[19],n2,Do_Not_Delete))){//pats[19] = "(edge > n3 < f)"

			  pf = assign(cons(getBindVal(lread("f")),getBindVal(lread("pf"))));
			  checkPath(n2,getBindVal(lread("n")),pf,getBindVal(lread("C")),getBindVal(lread("D")));
			  release(pf);
//			  pf = getBindVal(pats[2]); 					  //pats[2] = "pf"
//			  n3 = getBindVal(pats[18]);					  //pats[18] = "n3"
//			  if (vNull(pf)) {
//
//				if(!vNull(isDescInNode(pats[3],n3,Do_Not_Delete))){//pats[3] = "(atomic < C)"
//				  D = getBindVal(pats[4]);					  //pats[4] = "D"
//				  n = getBindVal(pats[5]);					  //pats[5] = "n"
//				  pushStack(car(currPD),D,n);
//				}
//				else {
//
//				  if(vNull(isDescInNode(pats[6],n3,Do_Not_Delete))){//pats[6] = "(not (atomic < C))"
//					tmp = assign(Build(pats[7]));			  //pats[7] = "(waitc (atomic < C) < n < D)"
//					addDescription(n3,tmp);
//					release(tmp);
//				  }
//				}
//			  }
//			  else {
//				putBindVal(pats[8],car(pf));				 //pats[8] = "g"
//				putBindVal(pats[9],cdr(pf));				 //pats[9] = "pftail"
//				tmp = assign(Build(pats[10]));				  //pats[10] = "(waitf < g < n << pftail (atomic < C) < D)"
//				addDescription(n3,tmp);
//				release(tmp);
//			  }
		  }
		  else addDescription(n2,car(iter1));
		}
		else addDescription(n2,car(iter1)); 				// any other description
	  }
	  rplacd(iter,cdr(cdr(iter)));
	}
	else {
	  putBindVal(lread("node1"),n1);
	  tmp = assign(Build(pats[30]));						//pats[30] = (copyQvar < node1)
	  addDescription(n2,tmp);
	  release(tmp);
	  iter = cdr(iter);
	}
  }
  release(tmpL);
  tmp = car(getprophead(n1, car(currPD)));
  rplacd(tmp,cons(cons(lread("eqclass"),cons(n2,getNil())),cdr(tmp)));

  for (iter1 = car(eqlist); !vNull(iter1) ; iter1 = cdr(cdr(iter1))) {
	n4 = car(iter1);
	n3 = car(cdr(iter1));
	if (!add_Equality(n3,n4)) {
	  release(eqlist);
	  return FALSE;
	}
  }
  release(eqlist);
  return TRUE;
}

boolean add_Equality (Lptr node1,Lptr node2) {

  Lptr eC1, eC2 ;
  static	int counter = 0;	// Avoids non-determinism

  if (!eq(node1,node2)) {
	if (exists_Ineqality_Edge(node1,node2))
	  return FALSE;
	eC1 = getEqClass(node1);
	eC2 = getEqClass(node2);
	if (!eq(eC1,eC2)) {
	  if(!vNull(isDescInNode(pats[16],eC1,Do_Not_Delete))) { //pats="(atomic QVAR)"
		if(!vNull(isDescInNode(pats[16],eC2,Do_Not_Delete))){//pats="(atomic QVAR)"
		  ++counter;
		  if (counter & 1) {
			if (!setEqClass(eC1, eC2)) return FALSE; 
		  } else {
			if (!setEqClass(eC2, eC1)) return FALSE; 
		  }
		}
		else {
		  {if (!setEqClass(eC1,eC2)) return FALSE ; }
		}
	  }
	  else	{
		if (!vNull(isDescInNode(pats[16],eC2,Do_Not_Delete))){//pats[16] ="(atomic QVAR)"
		  if (!setEqClass(eC2, eC1)) return FALSE;	  }
		else  {
		  ++counter;
		  if (counter & 1) {
		 
			if (!setEqClass(eC1, eC2)) return FALSE;
		  } else  {
			if (!setEqClass(eC2, eC1)) return FALSE;
		  }
		}
	  }
	}
  }
  return TRUE;
}

boolean add_Inequality (Lptr node1, Lptr node2) {
  Lptr tmp,E1,E2,tmp1,n4,n3,iter1;
  Lptr ineqlist;

  if (eq(getEqClass(node1),getEqClass(node2)))
	return FALSE;

  ineqlist = assign(cons(getNil(),getNil()));
  putBindVal(pats[11],node2);								//pats[11] = "n2"
  putBindVal(pats[13],node1);								//pats[13] = "n1"

  if (vNull(isDescInNode(pats[12],node1,Do_Not_Delete))) {	 //pats[12] = "(ineq < n2)"
	tmp = assign(Build(pats[12]));							//pats[12] = "(ineq < n2)"
	addDescription(node1,tmp);
	release(tmp);

	tmp = assign(Build(pats[14]));							//pats[14] = "(ineq < n1)"
	addDescription(node2,tmp);
	release(tmp);

//	  E1 = isDescInNode(pats[20],node1,Do_Not_Delete);		//pats[20] = "(redge > n3 > f)"
//	  while (!vNull(E1)){
//		E2 = isDescInNode(pats[21],node2,Do_Not_Delete);	//pats[21] = "(redge > n4 < f)"
//		while (!vNull(E2)) {
//		  add_Inequality(getBindVal(pats[18]),getBindVal(pats[22]));	 //18="n3"	22 ="n4"
//		  E2 = isDescInNode(pats[21],node2,Do_Not_Delete);	//pats[21] = "(redge > n4 < f)"
//		}
//		E1 = isDescInNode(pats[20],node1,Do_Not_Delete);	//pats[20] = "(redge > n3 > f)"
//
//	  }

/**** The following code is correct but is not effecient. It shud be changed *******/

	E1 = isDescInNode(pats[20],node1,Delete);
	while (!vNull(E1)) {
	  tmp1 = assign(lread("(egder < n3 < f)")); tmp = assign(Build(tmp1));
	  addDescription(node1,tmp);
	  release(tmp); release(tmp1);
	  E2 = isDescInNode(pats[21],node2,Delete);
	  while (!vNull(E2)){
		tmp1 = assign(lread("(egder < n4 < f)")); tmp = assign(Build(tmp1));
		addDescription(node2,tmp);
		release(tmp); release(tmp1);

		vaddList(ineqlist,getEqClass(getBindVal(pats[18])));
		vaddList(ineqlist,getEqClass(getBindVal(pats[22])));


		//add_Inequality(getEqClass(getBindVal(pats[18])),getEqClass(getBindVal(pats[22])));
		E2 = isDescInNode(pats[21],node2,Delete);
	  }
	  E1 = isDescInNode(pats[20],node1,Delete);
	}
	tmp1=assign(lread("(egder > n3 > f)")); tmp = assign(Build(tmp1));
	E1 = isDescInNode(tmp,node1,Delete);
	release(tmp); release(tmp1);
	while(!vNull(E1)) {
	  tmp1=assign(lread("(redge < n3 < f)")); tmp = assign(Build(tmp1));
	  addDescription(node1,tmp);
	  release(tmp); release(tmp1);
	  tmp1=assign(lread("(egder > n3 > f)")); tmp = assign(Build(tmp1));
	  E1 = isDescInNode(tmp,node1,Delete);
	  release(tmp); release(tmp1);
	}
	tmp1= assign(lread("(egder > n4 > f)")); tmp = assign(Build(tmp1));
	E1 = isDescInNode(tmp,node2,Delete);
	release(tmp); release(tmp1);
	while(!vNull(E1)) {
	  tmp1=assign(lread("(redge < n4 < f)")); tmp = assign(Build(tmp1));
	  addDescription(node2,tmp);
	  release(tmp); release(tmp1);
	  tmp1= assign(lread("(egder > n4 > f)")); tmp = assign(Build(tmp1));
	  E1 = isDescInNode(tmp,node2,Delete);
	  release(tmp); release(tmp1);
	}

	for (iter1 = car(ineqlist); !vNull(iter1) ; iter1 = cdr(cdr(iter1))) {
	  n4 = car(iter1);
	  n3 = car(cdr(iter1));
	  if (!add_Inequality(n3,n4)) {
		release(ineqlist);
		return FALSE;
	  }
	}

  }
  release(ineqlist);
  return TRUE;
}

Lptr findPath(Lptr fromNode,Lptr Path) {
  Lptr n, fA, ftail, N1;
  n = fromNode;
  if (!vNull(Path)) {
	fA = car(Path);
	ftail = cdr(Path);
	while (!vNull(fA)) {
	  N1 = findEdge(n,fA);
	  if (vNull(N1)) {
		if (eq(n,pats[23])) {								//pats[23] = "PDNDROOT"
		  N1 = createNode(fA);
		}
		else {
		  N1 = createNode(getNil());
		  add_Edge (n,N1,fA);
		}
	  }
	  n = N1;
	  fA = car(ftail);
	  ftail = cdr(ftail);
	}
  }
  return n;
}

Lptr findEdge(Lptr node, Lptr label) {
  Lptr desc,N1;
  if (eq(node,pats[23])) {									//pats[23] = "PDNDROOT"
	if (vNull(getprophead(label, car(currPD))))
	  return getNil();
	return label;
  }
  else {
	putBindVal(pats[24],label); 							//pats[24] = "lbl"
	desc = isDescInNode(pats[25],node,Do_Not_Delete);		//pats[25] = "(edge > n1 < lbl)"
	if (!vNull(desc)) {
	  N1 = getBindVal(pats[13]);							//pats[13] = "n1"
	  //return N1;
	  return getEqClass(N1);
	}
	else return getNil();
  }
}

void addDescription(Lptr Node,Lptr Desc) {
  Lptr tmp;
  if (Match(pats[16],Desc)){								//pats[16] = "(atomic QVAR)"
	tmp = car(getprophead(Node, car(currPD)));
	rplacd(tmp,cons(Desc,cdr(tmp)));
  }
  else {
	tmp = car(getprophead(getEqClass(Node), car(currPD))) ;
	rplacd(tmp ,cons(Desc,cdr(tmp)));
  }
}

// This function is not required -- instead use "isDescInNode" with "TRUE" as its last argument
void delDescription(Lptr Node,Lptr Desc) {
  Lptr Dlist, iterator, prev ;

  prev =  car(getprophead(Node, car(currPD)));
  Dlist = cdr(prev);
  for (iterator = Dlist ; !vNull(iterator) ; iterator = cdr(iterator),prev=cdr(prev)) {
	if (eq(Desc,car(iterator)))
		prev = rplacd(prev,cdr(iterator));
  }

  prev = car(getprophead(getEqClass(Node), car(currPD))) ;
  Dlist = cdr(prev);
  for (iterator = Dlist ; !vNull(iterator) ; iterator = cdr(iterator),prev=cdr(prev)) {
	if (eq(Desc,car(iterator)))
		prev = rplacd(prev,cdr(iterator));
  }
}

Lptr getEqClass(Lptr Node){
  Lptr desclist,iter,N1;
  desclist = cdr(car(getprophead(Node, car(currPD))));
  for (iter = desclist ; !vNull(iter) ; iter = cdr(iter)) {
	if (Match(pats[26],car(iter))) {					   //pats[26] = "(eqclass > N1)"
	  N1 = getBindVal(pats[27]);							//pats[27] = "N1"
	  return getEqClass(N1);
	}
  }
  return Node;
}

Lptr getRHS (Lptr LHS) {
  Lptr globalRHS = getNil(), localRHS = getNil();
  boolean nodeInCurrPD = FALSE;

  if (!nullEval(LHS)) {
	globalRHS = getprop(LHS,pats[28]);						// pats[28] = "GlobalT"
	nodeInCurrPD = !vNull(getprophead(LHS,car(currPD)));
	if (nodeInCurrPD)
	  localRHS = car(getprop(LHS,car(currPD)));

	if (vNull(localRHS) && !vNull(globalRHS))
	  return globalRHS;
	if (!vNull(localRHS) && vNull(globalRHS))
	  return localRHS;
	if (!vNull(localRHS) && !vNull(globalRHS))
	  return(cons(pats[29],cons(globalRHS,cons(localRHS,getNil()))));//pats[29] = "and"
  }
  return getNil();
}

/* Following functions returns a description/NULL depending on whether
   the given pattern matches any description in the given node or not.
   =======================================================================	 
*/
Lptr isDescInNode (Lptr Pat, Lptr node, boolean deletePat) {
  //Note : "node" is an atom.
  Lptr Dlist, iterator, prev,temp;

  /* process the node */
  prev =  car(getprophead(node,car(getCurrPD())));
  Dlist = cdr(prev);
  for (iterator = Dlist ; !vNull(iterator) ; iterator = cdr(iterator),prev=cdr(prev)) {
	if (Match(Pat,car(iterator))){
	  temp = car(iterator);
	  if (deletePat)
		prev = rplacd(prev,cdr(iterator));
	  return temp;
	}
  }

  /* process the eqClass node */
  prev = car(getprophead(getEqClass(node),car(getCurrPD()))) ;
  Dlist = cdr(prev);
  for (iterator = Dlist ; !vNull(iterator) ; iterator = cdr(iterator),prev=cdr(prev)) {
	if (Match(Pat,car(iterator))) {
	  temp = car(iterator);
	  if (deletePat)
		prev = rplacd(prev,cdr(iterator));
	  return temp;
	}
  }
  return getNil();
}


/*****Functions for manipulation of Stacks ********/

boolean isEmptyStack (Lptr stackHead) {
	if (vNull(car(stackHead)))
	  return TRUE;
	return FALSE;
}

Lptr pushStack (Lptr stackHead,Lptr Desc,Lptr name) {
   Lptr element;
   element = cons(Desc,cons(name,getNil()));
   vaddList(stackHead,element);
   return stackHead;
}

/* pop the element from the stack 
   return NULL if it is empty
*/
Lptr popStack(Lptr stackHead) {
  if (!isEmptyStack(stackHead)) {
	Lptr temp= assign(car(stackHead));
	rplaca(stackHead,cdr(temp));
	rplacd(temp,getNil());
	decrement(temp);
	return temp;
  }
  else{
	return NULL;
  }
}

/**************************************************************************************
 Expand Functions expand1() and expand2()
 They are the core of the CGU
 **************************************************************************************/

extern Lptr currPD ;
Lptr descList=NULL, tmpPatterns[10], tmpabcd=NULL;

/* the DLFDA constructors, see p.33 of Alex's master thesis.
   so far no key descriptions.
*/
char * descriptions =
  "("
	" (top)"											//1
	" (bottom)" 										//2
	" (atomic > C)" 									//3
	" (not (atomic > C))"								//4
	" (! or (forall > Pf (not (atomic > C))) > D2)" 	//5
	" (and > D1 > D2)"									//6
	" (equal (>* flist1 > f1)(>* flist2 > f2))" 		//7
	" (equal (>* flist > f1) ())"						//8
	" (equal () (>* flist > f1))"						//9
	" (nequal > Pf1 > Pf2)" 							//10
	" (at > D1 > f)"									//11
	" (forall > f > D1)"								//12
	" (! or > D1 > D2)" 								//13
  ")"  ;
/* More explanations:
  1: everything
  2: nothing
  3: primitive concept
  4: concept negation
  5: rule-based disjunction
  6: concept conjuction
  7: path agreement flist1.f1 = flist2.f2
  8: path agreement filist1.f1 = id
  9: path agreement id = flist1.f1	
 10: path disagreement
 11: inverse attributes D1@f
 12: attribute typing
 13: concept disjunction
*/

/* Descriptions added to a node during expansion.
   waitc and waitf are triggers waiting for a concept or path to be added.
   See Alex's master thesis p48-51
 */    
char * snrettap =
  "("
	" (not (atomic < C))"								//1
	" (waitc (atomic < C) > n1 > D2)"					//2
	" (atomic < C)" 									//3
	" (waitc (atomic < C) > N1 > D2)"					//4
	" (waitc (atomic < C) < N < D2)"					//5
	" (> f >* flist)"									//6
	" (waitf < f < N (<< flist) (atomic < C) < D2)" 	//7
	" (waitf < f < N (<< flist) (atomic < C) < D2)" 	//8
	" (equal (<< flist < f1) ())"						//9
  ")";

/* Initialize the patterns of the descriptions
   Using global variables:
	 descList, tmpabcd, tmpPatterns, descriptions, snrettap
*/	
void initPatterns(void) {
  int i;
  Lptr iter;
  if (descList==NULL ) {
	/* DLDFA description list */
	descList = assign(lread(descriptions));

	/* expansion description list */
	tmpabcd = assign(lread(snrettap));

	/* assign expansion descriptions to tmpPatterns */	  
	tmpPatterns[0] = getNil();
	for (i = 1,iter=tmpabcd; i<=9; iter=cdr(iter),i++)
	  tmpPatterns[i] = car(iter);
  }
}

/* Release the lists
   Using global variables: descList, tmpabcd */
void finalizePatterns(void) {
  if (descList != NULL){
	release(descList);
	release(tmpabcd);
	descList=NULL;
	tmpabcd=NULL;
  }
}

/* Match the description with those types of DLFDA
   ================================================
   Parameter: desc is the input description
   ================================================
   Returns: the index of the description type
*/	 
int getDescType(Lptr desc) {
	int i;
	Lptr descListIterator;

	/* Check the description by matching those in the DLFDA description list */
	for (i=1,descListIterator=descList ; i<=13 ; i++,descListIterator = cdr(descListIterator)) {
	  if (Match(car(descListIterator),desc))
		  return i;
	}
	printf("ERROR: Description does not match with any of the standard patterns.");
	printS(0, desc);
	printf("\n");
	return ERR_DESC_NOT_MATCH;
}


/* One of two core expansions of common graph utility
   expand1 is the deterministic expansion procedure that deals with all
   DLFDA constructors except general concept disjunction and the attribute
   type (the two are handled by expand2).
   See Alex's master's thesis p.47-50
*/	 
boolean expand1(void) {

  int descType;
  Lptr stackElement,D,N,tmp;
  Lptr D1,D2,N1,N2,N3,N4,PF,F1,F2,PF1,PF2;
  Lptr currStack1 = cdr(cdr(currPD)) ;			  //Head of stack1
  boolean N1_root = FALSE , N2_root = FALSE;

  initVars();
  initPatterns();
  pushMatchVar();

  while (!isEmptyStack(currStack1)) {
   stackElement = assign(popStack(currStack1));
   D = car(car(stackElement));					  //D is description
   N = car(cdr(car(stackElement))); 			 //N is node

   descType = getDescType(D);

   switch (descType) {
	 case TOP : 											//D = top
	   break;

	 case BOTTOM :											//D = bottom
	   release(stackElement);
	   popMatchVar();
	   return FALSE ;
	   break;

	 case Atomic_C :										//D = "(atomic C)"
		if (!vNull(isDescInNode(tmpPatterns[1],N,Do_Not_Delete))){	//tmpPatterns[1]="(not (atomic < C))"
		  release(stackElement);
		  popMatchVar();
		  return FALSE;
		}

		if (vNull(isDescInNode(D,N,Do_Not_Delete))) {		 //Note: D=(atomic C)
		  addDescription(N,D);
		  D1 = getRHS(getBindVal(lread("C")));
		  if (!vNull(D1))
			pushStack(currStack1,D1,N);

		  D1 = isDescInNode(tmpPatterns[2],N,Delete);			   //tmpPatterns[2]="(waitc (atomic < C) > n1 > D2)"
		  while (!vNull(D1)) {
			D2 = getBindVal(lread("D2"));
			N1 = getBindVal(lread("n1"));
			pushStack(currStack1,D2,N1);
			D1 = isDescInNode(tmpPatterns[2],N,Delete); 		   //tmpPatterns[2]="(waitc (atomic < C) > n1 > D2)"
		  }
		}
	   break;

	 case Not_Atomic_C :									//D = "(not (atomic C))"
		if (!vNull(isDescInNode(tmpPatterns[3],N,Do_Not_Delete))) { //tmpPatterns[3]="(atomic < C)"
		  release(stackElement);
		  popMatchVar();
		  return FALSE;
		}

		D1 = isDescInNode(tmpPatterns[4],N,Delete); 			   //tmpPatterns[4]="(waitc (atomic < C) > N1 > D2)"
		while (!vNull(D1))
		  D1 = isDescInNode(tmpPatterns[4],N,Delete);			   //tmpPatterns[4]="(waitc (atomic < C) > N1 > D2)"

		addDescription(N,D);
	   break;
															//D ="(or  (forall Pf (not (atomic C))) D2)"
	 case NFPD :
		  checkPath(N,N,getBindVal(lread("Pf")),getBindVal(lread("C")),getBindVal(lread("D2")));
//			Flist = getBindVal(lread("Pf"));
//			Ni = N;
//			do {
//			  if (vNull(Flist)) {
//				if (!vNull(isDescInNode(tmpPatterns[3],Ni,Do_Not_Delete)))	 //tmpPatterns[3]="(atomic < C)"
//				  pushStack(currStack1,getBindVal(lread("D2")),Ni);
//				else if (vNull(isDescInNode(tmpPatterns[1],Ni,Do_Not_Delete))) {  //tmpPatterns[1]="(not (atomic < C))"
//				  putBindVal(lread("N"),N);
//				  tmp = assign(Build(tmpPatterns[5]));				 //tmpPatterns[5]="(waitc (atomic < C) < N < D2)"
//				  addDescription(Ni,tmp);
//				  release(tmp);
//				}
//				F = getNil();
//			  }
//			  else {
//				F = car(Flist);
//				Flist = cdr(Flist);
//				N1 = findEdge(Ni,F);
//				if (vNull(N1)) {				// if the edge does not exist
//				  boolean isroot;
//				  isroot = eq(Ni,lread("PDNDROOT"));
//				  if (isroot) {
//					Ni = createNode(F);
//					//Flist = cdr(Flist);
//				  }
//				  else {
//					putBindVal(lread("N"),N);
//					putBindVal(lread("f"),F);
//					putBindVal(lread("flist"),Flist);
//					tmp = assign(Build(tmpPatterns[8]));			   //tmpPatterns[8]="(waitf < f < N (<< flist) (atomic < C) < D2)"
//					addDescription(Ni,tmp);
//					release(tmp);
//					F = getNil();
//				  }
//				}
//				else {
//				  //Flist = cdr(Flist);
//				  Ni = N1;
//				}
//			  }
//			} while (!vNull(F));
	   break;


	 case And_D1_D2 :										//D = "(and D1 D2)"
		D2 = getBindVal(lread("D2"));
		pushStack(currStack1,D2,N);
		D1 = getBindVal(lread("D1"));
		pushStack(currStack1,D1,N);
	   break;

	 case Equal_Path :										//D = "(equal (flist1 f1)(flist2 f2))"
		PF1 = getBindVal(lread("flist1"));
		PF2 = getBindVal(lread("flist2"));
		F1 = getBindVal(lread("f1"));
		F2 = getBindVal(lread("f2"));
		N1 = findPath(N,PF1);
		N2 = findPath(N,PF2);
		N3 = findEdge(N1,F1);
		N4 = findEdge(N2,F2);
		N1_root = eq(N1,lread("PDNDROOT"));
		N2_root = eq(N2,lread("PDNDROOT"));

		if (vNull(N3) && N1_root)
		  N3 = createNode(F1);
		if (vNull(N4) && N2_root)
		  N4 = createNode(F2);
		if (vNull(N3) && vNull(N4) && !N1_root && !N2_root) {
		  N3 = createNode(getNil());
		  add_Edge(N1,N3,F1);
		}
		if (vNull(N4) && !N2_root)
		  add_Edge(N2,N3,F2);
		if ((!vNull(N3) && !vNull(N4)) || (N1_root && N2_root) || (!vNull(N3) && !N1_root && N2_root) || (vNull(N3) && !vNull(N4) && N1_root) ) {
		  if(!add_Equality(N3,N4)) {
			release(stackElement);
			popMatchVar();
			return FALSE;
		  }
		}
		if (vNull(N3) && !N1_root && (!vNull(N4) || N2_root))
		  add_Edge(N1,N4,F1);

	   break;

	 case Path_Id1 :										//D = "(equal (flist f1) ())"
		PF = getBindVal(lread("flist"));
		F1 = getBindVal(lread("f1"));
		N1 = findPath(N,PF);
		N2 = findEdge(N1,F1);
		if (!vNull(N2)) {
		  if (!add_Equality(N2,N)) {
			release(stackElement);
			popMatchVar();
			return FALSE;
		  }
		}
		else
		  add_Edge(N1,N,F1);
	   break;

	 case Path_Id2 :										//D = "(equal () (flist f1))"
		tmp = assign(Build(tmpPatterns[9]));					   //tmpPatterns[9]="(equal (<< flist < f1) ())"
		pushStack(currStack1,tmp,N);
		release(tmp);
	   break;

	 case Nequal_Paths: 									//D = "(nequal Pf1 Pf2)"
		 PF1 = getBindVal(lread("Pf1"));
		 PF2 = getBindVal(lread("Pf2"));
		 N1 = findPath(N,PF1);
		 N2 = findPath(N,PF2);
		 if (!add_Inequality(N1,N2)) {
		   release(stackElement);
		   popMatchVar();
		   return FALSE;
		 }
	   break;

	 case DatF: 											//D = "(at D1 f)"
		F1 = getBindVal(lread("f"));
		D1 = getBindVal(lread("D1"));
		N1 = createNode(getNil());
		add_Edge(N1,N,F1);
		pushStack(currStack1,D1,N1);
	   break;

	 case Forall_F_D1:										//D = "(forall f D1)"
		N1 = findEdge(N,getBindVal(lread("f")));
		if (!vNull(N1))
		  pushStack(currStack1,getBindVal(lread("D1")),N1);
		else  {
		  if (eq(N,lread("PDNDROOT"))) {
			N1 = createNode(getBindVal(lread("f")));
			pushStack(currStack1,getBindVal(lread("D1")),N1);
		  }
		  else
			pushStack(cdr(currStack1),D,N);
		}
	   break;

	 case Or_D1_D2: 										//D = "(or D1 D2)"
		pushStack(cdr(currStack1),D,N);
	   break;
	 }
	 release(stackElement);
  }
  popMatchVar();
  return TRUE;
}

boolean expand2(void) {

  Lptr stackElement,D,N,pats,N1;
  Lptr currStack1 = cdr(cdr(currPD)) ;					  //Head of stack1
  Lptr currStack2 = cdr(currStack1);					   //Head of stack2

  initVars();
  pushMatchVar();

  if (!isEmptyStack(currStack2)) {
	stackElement = assign(popStack(currStack2));
	D = car(car(stackElement)); 						  //D is description
	N = car(cdr(car(stackElement)));					 //N is node

	pats = assign(lread("((forall > f > D1) (! or > D1 > D2))"));

	if (Match(car(pats),D)) {							   //D="(forall > f > D1)"
	  N1 = findEdge(N,getBindVal(lread("f")));
	  if (!vNull(N1))
		pushStack(currStack1,getBindVal(lread("D1")),N1);
	  else {
		N1= createNode(getNil());
		add_Edge(N,N1,getBindVal(lread("f")));
		pushStack(currStack1,getBindVal(lread("D1")),N1);
	  }
	  if (expand1()) {
		release(stackElement);
		release(pats);
		popMatchVar();
		return expand2();
	  }
	  else {
		release(stackElement);
		release(pats);
		popMatchVar();
		return FALSE;
	  }
	}
	else if (Match(car(cdr(pats)),D)) { 				  //D="(! or > D1 > D2)"
	  beginVer();
	  pushStack(currStack1,getBindVal(lread("D1")),N);
	  if (expand1()) {
		if (expand2()) {
		  release(stackElement);
		  release(pats);
		  //abortVer();
		  popMatchVar();
		  return TRUE;
		}
	  }
	  abortVer();
	  beginVer();
	  pushStack(currStack1,getBindVal(lread("D2")),N);
	  if (expand1()) {
		if (expand2()) {
		  release(stackElement);
		  release(pats);
		  //abortVer();
		  popMatchVar();
		  return TRUE;
		}
	  }
	  release(stackElement);
	  release(pats);
	  abortVer();
	  popMatchVar();
	  return FALSE;
	}
  }
  else {
	popMatchVar();
	return TRUE;
  }
  assert(0);
  return FALSE;
}

void checkPath(Lptr Ni,Lptr N,Lptr Flist,Lptr C,Lptr D2){
  Lptr tmp,N1,F;
  Lptr currStack1 = cdr(cdr(getCurrPD()));

  initVars();
  initPatterns();
  pushMatchVar();

  putBindVal(lread("N"),N);
  putBindVal(lread("C"),C);
  putBindVal(lread("D2"),D2);

  do {
	if (vNull(Flist)) {
	  if (!vNull(isDescInNode(tmpPatterns[3],Ni,Do_Not_Delete)))	//tmpPatterns[3]="(atomic < C)"
		pushStack(currStack1,D2,Ni);
	  else if (vNull(isDescInNode(tmpPatterns[1],Ni,Do_Not_Delete))) {	//tmpPatterns[1]="(not (atomic < C))"
		tmp = assign(Build(tmpPatterns[5]));					   //tmpPatterns[5]="(waitc (atomic < C) < N < D2)"
		addDescription(Ni,tmp);
		release(tmp);
	  }
	  F = getNil();
	}
	else {
	  F = car(Flist);
	  Flist = cdr(Flist);
	  N1 = findEdge(Ni,F);
	  if (vNull(N1)) {										 // if the edge does not exist
		if (eq(Ni,lread("PDNDROOT"))) {
		  Ni = createNode(F);
		  //Flist = cdr(Flist);
		}
		else {
		  putBindVal(lread("f"),F);
		  putBindVal(lread("flist"),Flist);
		  tmp = assign(Build(tmpPatterns[8]));					   //tmpPatterns[8]="(waitf < f < N (<< flist) (atomic < C) < D2)"
		  addDescription(Ni,tmp);
		  release(tmp);
		  F = getNil();
		}
	  }
	  else {
		//Flist = cdr(Flist);
		Ni = N1;
	  }
	}
  } while (!vNull(F));
  popMatchVar();
}

