#include <algorithm>
#include <fstream>
#include <iosfwd>

#include <GL/gl.h>
#include <GL/glu.h>

#include "mesh.h"

// A Triple holds the numbers given for a vertex when describing a
// face.  We care about the first number (an index into the array
// of vertices) and the third number (an index into the array of 
// normals).
struct Triple {
	int arr_[ 3 ];

	Triple()
	{
		arr_[0] = 0;
		arr_[1] = 0;
		arr_[2] = 0;
	}

	int& operator[]( size_t idx )
	{
		return arr_[ idx ];
	}
	int operator[]( size_t idx ) const
	{
		return arr_[ idx ];
	}
};

// A description of a half-edge in the mesh.  During reading,
// we'll build a big collection of these and use them to determine
// connectivity between faces later.
struct he_desc {
	size_t 	v1_;
	size_t 	v2_;

	size_t  tri_;  // Which triangle is using this edge?
	size_t 	index_; // At which vertex of the triangle does this edge start?

	he_desc( size_t v1, size_t v2, size_t t, size_t i )
		: v1_( v1 )
		, v2_( v2 )
		, tri_( t )
		, index_( i )
	{
		if( v1_ > v2_ ) {
			std::swap( v1_, v2_ );
		}
	}
};

bool he_comp( const he_desc& a, const he_desc& b )
{
	if( a.v1_ < b.v1_ ) {
		return true;
	} else if( a.v1_ == b.v1_ ) {
		return a.v2_ < b.v2_;
	} else {
		return false;
	}
}

// Return a list of triples stored in a vector.  Each triple is of
// the form idx[0]/idx[1]/idx[2].  idx[0] is an index into the array
// of vertices.  idx[2] is an index into the array of normals.  Don't
// worry about idx[1].

static void readFace( std::istream& is, std::vector<Triple>& triples )
{
	char ch;
	while( is ) {
		ch = is.get();
		if( ch >= '0' && ch <= '9' ) {
			int i = 0;
			Triple entry;
			entry[0] = ch - '0';

			while ((ch = is.get()) != ' ') {
				if (ch >= '0' && ch <= '9') {
					entry[i] = entry[i]*10 + (ch - '0');
				} else if (ch == '/') {
					++i;
				} else if (ch == '\n') {
					--entry[0];
					--entry[1];
					--entry[2];
					triples.push_back(entry);
					return;
				} else {
					if (!is) return;
					// Bad face input
				}
			}
			--entry[0];
			--entry[1];
			--entry[2];
			triples.push_back(entry);
		} else if( ch != ' ' && ch != '\t' ) {
			if( !is ) {
				return;
			}
			// Bad face input
		}
	}
}

static bool addFace( 
	const Triple& t1, const Triple& t2, const Triple& t3, 
	std::vector<Tri*>& tris_, const std::vector<Vertex*>& verts_,
	std::vector<Vector3D>& normals, std::vector<he_desc>& edges )

{
	Tri *tri = new Tri;

	bool ret = false;

	int vs[3] = { t1[0], t2[0], t3[0] };
	int ns[3] = { t1[2], t2[2], t3[2] };

	for( size_t v = 0; v < 3; ++v ) {
		tri->vs_[v] = verts_[ vs[v] ];

		if( ns[v] >= 0 ) {
			tri->ns_[v] = normals[ ns[v] ];
		} else {
			ret = true;
		}
	}

	Vector3D v01 = tri->vs_[1]->pos_ - tri->vs_[0]->pos_;
	Vector3D v02 = tri->vs_[2]->pos_ - tri->vs_[0]->pos_;

	// Just in case, compute the face normal and add it to 
	// the fake normal for each vertex.

	tri->normal_ = v01.cross( v02 );
	tri->normal_.normalize();

	for( size_t v = 0; v < 3; ++v ) {
		tri->vs_[ v ]->fake_normal_ = 
			tri->vs_[ v ]->fake_normal_ + tri->normal_;
	}

	size_t ti = tris_.size();
	tris_.push_back( tri );

	edges.push_back( he_desc( vs[0], vs[1], ti, 0 ) );
	edges.push_back( he_desc( vs[1], vs[2], ti, 1 ) );
	edges.push_back( he_desc( vs[2], vs[0], ti, 2 ) );

	return ret;
}

TriMesh::TriMesh( std::istream& is )
{
	std::vector<Vector3D> 	normals;
	std::vector<he_desc> 	edges;

	// You might want to experiment with setting this to true,
	// which will cause this (somewhat flaky) function to always
	// invent vertex normals from face normals.
	bool do_normals = false;

	char ch = 0;
	while( is ) {
		is >> std::ws >> ch;
		if( !is ) {
			// TODO: Check for error conditions.
			break; 
		}

		switch( ch ) {
			case 'v': {
				ch = is.get();
				switch (ch) {
				case ' ': {
					// Vertex
					double x, y, z;
					is >> x >> y >> z;
					Point3D vert( x, y, z );
					verts_.push_back( new Vertex( vert ) );
					break; }
				case 'n': {
					// Normal
					double x, y, z;
					is >> x >> y >> z;
					Vector3D new_normal( x, y, z );
					new_normal.normalize();
					normals.push_back( new_normal );
					break; }
				}
				while( is && is.get() != '\n'); // Ignore rest of line
				break; }
			case 'f': { // Face
				std::vector<Triple> triples;
				readFace( is, triples );

				if (triples.size() == 3) {
					if( addFace( triples[0], triples[1], triples[2], 
							tris_, verts_, normals, edges ) ) {
						do_normals = true;
					}
				} else {
					// Not a triangle. Assume simple convex polygon,
					// triangulate by ear cutting (draw an edge from
					// the first vertex to all others).
					for( size_t i = 2; i < triples.size(); ++i ) {
						if( addFace( triples[0], triples[i-1], triples[i], 
								tris_, verts_, normals, edges ) ) {
							do_normals = true;
						}
					}
				}
				break; }
			// Ignore all of these potential objects.  You may want
			// to take some of these into account (say, grouping).
			case '#': // Comment
			case '$': // Apparently this is also used for comments sometimes
			case 'l': // Line, ignore
			case 'p': // Point, ignore
			case 'm': // material library, ignore
			case 'g': // group, ignore
			case 's': // smoothing group, ignore
			case 'u': // material, ignore
			case 't': // Texture coordinate
			case 'r': // tangent
			default: // anything else we ignore as well
				while (is && is.get() != '\n') ; // Ignore rest of line
				break;
		}
	}

	for( size_t idx = 0; idx < verts_.size(); ++idx ) {
		verts_[ idx ]->fake_normal_.normalize();
	}

	// If the previous loop didn't come across any vertex normals,
	// assume that we have to make them up by averaging face normals
	// for all faces at a vertex.  Assign those normals to vertices
	// here.  (If any vertex normals were detected, don't do this,
	// and treat edges with mismatched normals as creases in the model).

	if( do_normals ) {
		for( size_t idx = 0; idx < tris_.size(); ++idx ) {
			Tri *tri = tris_[ idx ];

			for( size_t v = 0; v < 3; ++v ) {
				tri->ns_[ v ] = tri->vs_[ v ]->fake_normal_;
			}
		}
	}

	// Now derive adjacency information.  Find creases while you're at 
	// it.  We sort edge references by pairs of indices into the vertex
	// array.  If an edge is used twice (by the two faces on either side
	// of it), this step should put those two uses next to one another.
	// Otherwise our mesh has boundary curves and we treat those as
	// creases.

	std::sort( edges.begin(), edges.end(), he_comp );

	size_t idx = 0;
	while( idx < edges.size() ) {
		const he_desc& he1 = edges[idx];
		const he_desc& he2 = edges[idx+1];

		if( (he1.v1_ != he2.v1_) || (he1.v2_ != he2.v2_) ) {
			std::cerr << "Warning! unmatched half edge" << std::endl;
			// These two half_edges weren't a match, try jumping
			// ahead by one.
			++idx;
		} else {
			// They were a match, jump ahead by two.
			idx += 2;
		}

		Tri *t1 = tris_[ he1.tri_ ];
		Tri *t2 = tris_[ he2.tri_ ];

		Edge *edge = new Edge;

		edge->v1_ = verts_[ he1.v1_ ];
		edge->v2_ = verts_[ he1.v2_ ];

/*
		std::cerr << "v1 = " << (void*)edge->v1_ << std::endl;
		std::cerr << "v2 = " << (void*)edge->v2_ << std::endl;
		std::cerr << "t1: " 
				  << (void*)t1->vs_[he1.index_] << " --> "
				  << (void*)t1->vs_[(he1.index_+1)%3] << std::endl;
		std::cerr << "t2: " 
				  << (void*)t2->vs_[he2.index_] << " --> "
				  << (void*)t2->vs_[(he2.index_+1)%3] << std::endl;
*/
		
		if( t1->vs_[ he1.index_ ] == edge->v1_ ) {
			edge->t1_ = t1;
			edge->e1_ = he1.index_;
			t1->es_[ he1.index_ ] = edge;

			if( t2->vs_[ he2.index_ ] != edge->v2_ ) {
				std::cerr << "Warning -- edge direction mismatch 1." 
						  << std::endl;
			}

			edge->t2_ = t2;
			edge->e2_ = he2.index_;
			t2->es_[ he2.index_ ] = edge;
		} else {
			edge->t2_ = t1;
			edge->e2_ = he1.index_;
			t2->es_[ he2.index_ ] = edge;

			if( t2->vs_[ he2.index_ ] != edge->v1_ ) {
				std::cerr << "Warning -- edge direction mismatch 2." 
						  << std::endl;
			}

			edge->t1_ = t2;
			edge->e1_ = he2.index_;
			t1->es_[ he1.index_ ] = edge;
		}

		edges_.push_back( edge );
		
		Vector3D n1 = edge->t1_->ns_[ edge->e1_ ];
		Vector3D n2 = edge->t2_->ns_[ (edge->e2_+1)%3 ];

		if( n1.dot( n2 ) < (1.0-1e-4) ) {
			// Two normals associated with this vertex disagree.
			// You can either overwrite with the inferred vertex
			// normal (commented out) or record this edge as a 
			// crease.
		/*
			edge->t1_->ns_[ edge->e1_ ] = fake_normals[ t1.vs[ v1p ] ];
			t2.ns[ v2p ] = fake_normals[ t2.vs[ v2p ] ];
		*/
			creases_.push_back( edge );
		} else {
			n1 = edge->t1_->ns_[ (edge->e1_+1)%3 ];
			n2 = edge->t2_->ns_[ edge->e2_ ];

			if( n1.dot( n2 ) < (1.0-1e-4) ) {
				creases_.push_back( edge );
			/*
				edge->t1_->ns_[ edge->e1_ ] = fake_normals[ t1.vs[ v1p ] ];
				t2.ns[ v2p ] = fake_normals[ t2.vs[ v2p ] ];
			*/
			}
		}
	}
}
