/****************************************************************************
 *
 * CS488 -- Introduction to Computer Graphics
 *
 * grmodule.c
 *
 * Glue code that creates Python bindings for C data types and functions. 
 *
 * University of Waterloo Computer Graphics Lab / 2003
 *
 ****************************************************************************/

#include <Python.h>

#include "algebra.h"
#include "gr.h"

struct PyMatrixObject {
    PyObject_HEAD
	double data[16];
};

static PyTypeObject *pMatrix_Type = NULL;

/****************************************************************************
 *
 * Define a Python binding to the GrMatrix4x4 type that lets us create
 * them and pass them around conveniently at the script level.
 *
 ****************************************************************************/

static PyMatrixObject *new_matrix( const Matrix4x4 M )
{ 
	/* Implicitly creates an 'owned' reference, so no need to 
	   use Py_INCREF here. */
	PyMatrixObject *ret = PyObject_New( PyMatrixObject, pMatrix_Type );

	std::copy( M.begin(), M.end(), ret->data );
	return ret;
}

static void matrix_dealloc( PyObject* self )
{
    PyObject_Del( self );
}

static PyObject * matrix_str( PyMatrixObject *obj )
{
	char buffer[ 1024 ];
	sprintf( buffer, 
		"[%f %f %f %f]\n[%f %f %f %f]\n[%f %f %f %f]\n[%f %f %f %f]", 

		obj->data[0], 
		obj->data[1], 
		obj->data[2], 
		obj->data[3], 

		obj->data[4], 
		obj->data[5], 
		obj->data[6], 
		obj->data[7], 

		obj->data[8], 
		obj->data[9], 
		obj->data[10], 
		obj->data[11], 

		obj->data[12], 
		obj->data[13], 
		obj->data[14], 
		obj->data[15] );

    return PyString_FromString( buffer );
}

/*
 * We might as well make the matrix's elements accessible in Python
 * using the Sequence protocol.  This way, matrices look a little
 * bit like lists of 16 floats.
 */

/* Allows you to do 'len( M )' for a matrix M in Python */
static int matrix_length( PyMatrixObject *obj )
{
	return 16;
}

/* Allows you to do 'x = M[12]' in Python */
static PyObject *matrix_item( PyMatrixObject *a, int i )
{
	if( (i >= 0) && (i < 16) ) {
		return PyFloat_FromDouble( a->data[ i ] );
	} else {
        PyErr_SetString( PyExc_IndexError, "matrix index out of range" );
        return NULL;
	}
}

/* Allows you to do 'M[12] = x' in Python */
static int matrix_ass_item( PyMatrixObject *a, int i, PyObject *v )
{
	double d;

	if( (i >= 0) && (i < 16) ) {
		if( !PyFloat_Check( v ) ) {
			return -1;
		}
		d = PyFloat_AsDouble( v );
		a->data[ i ] = d;
		return 0;
	} else {
        PyErr_SetString( PyExc_IndexError, 
			"matrix assignment index out of range");
        return -1;
    }
}

/* This table of pointers to functions defines the ways that our
   matrix type will behave like a sequence.  We only define three
   functions -- len( M ), M[i], and M[i] = v. */
static PySequenceMethods Matrix_Sequence = {
	(inquiry)matrix_length, 		/* inquiry sq_length; */ 
	0, 								/* binaryfunc sq_concat; */
	0,								/* intargfunc sq_repeat; */
	(intargfunc)matrix_item,		/* intargfunc sq_item; */
	0,								/* intintargfunc sq_slice; */
	(intobjargproc)matrix_ass_item,	/* intobjargproc sq_ass_item; */
	0,								/* intintobjargproc sq_ass_slice; */
	0, 								/* objobjproc sq_contains; */
	0,								/* binaryfunc sq_inplace_concat; */
	0,								/* intargfunc sq_inplace_repeat; */
};

static PyObject *matrix_mult( PyMatrixObject *a, PyMatrixObject *b )
{
	PyMatrixObject *ret;

	ret = new_matrix( Matrix4x4( a->data ) * Matrix4x4( b->data ) );
	return (PyObject*)ret;
}

static PyObject *matrix_inv( PyMatrixObject *a )
{
	PyMatrixObject *ret;

	ret = new_matrix( Matrix4x4( a->data ).invert() );
	return (PyObject*)ret;
}

/* We also make our matrices look a bit like numbers.  We overload
   the binary * operator to do matrix multiplication and the unary ~ 
   operator to do matrix inversion.  */
static PyNumberMethods Matrix_Number = {
	0, 						/* nb_add */
	0, 						/* nb_subtract */
	(binaryfunc)matrix_mult, /* nb_multiply */
    0, 						/* nb_divide */
    0,    					/* nb_remainder */
    0, 						/* nb_divmod */
    0,   					/* nb_power */
    0, 						/* nb_negative */
    0, 						/* nb_positive */
    0, 						/* nb_absolute */
    0,   					/* nb_nonzero */
    (unaryfunc)matrix_inv,	/* nb_invert */
    0, 						/* nb_lshift */
    0, 						/* nb_rshift */
    0,    					/* nb_and */
    0,    					/* nb_xor */
    0, 						/* nb_or */
    0,          			/* nb_coerce */
    0, 						/* nb_int */
    0,    					/* nb_long */
    0,   					/* nb_float */
    0, 						/* nb_oct */
    0,     					/* nb_hex */
    0,          			/* nb_inplace_add */
    0,          			/* nb_inplace_subtract */
    0,          			/* nb_inplace_multiply */
    0,          			/* nb_inplace_divide */
    0,          			/* nb_inplace_remainder */
    0,          			/* nb_inplace_power */
    0,          			/* nb_inplace_lshift */
    0,          			/* nb_inplace_rshift */
    0,          			/* nb_inplace_and */
    0,          			/* nb_inplace_xor */
    0,          			/* nb_inplace_or */
};

/* This table defines the core of the Matrix type in Python. */
static PyTypeObject Matrix_Type = {
    PyObject_HEAD_INIT(NULL)
    0,
    "matrix",
    sizeof( PyMatrixObject ),
    0,

    (destructor)matrix_dealloc, /* tp_dealloc */
    0,          				/* tp_print */
    0,          				/* tp_getattr */
    0,          				/* tp_setattr */
    0,          				/* tp_compare */
    0,          				/* tp_repr */
    &Matrix_Number, 			/* tp_as_number */
    &Matrix_Sequence,			/* tp_as_sequence */
    0,          				/* tp_as_mapping */
    0,          				/* tp_hash */
    0,          				/* tp_call */
    (reprfunc)matrix_str    	/* tp_str */

	/* Other fields default to 0 */
};

static PyObject *gr_new_identity_cmd( PyObject *self, PyObject *args )
{
	if( !PyArg_ParseTuple( args, "" ) ) {
		return NULL;
	}

	return (PyObject *)new_matrix( Matrix4x4() );
}

static PyObject *gr_new_rotation_cmd( PyObject *self, PyObject *args )
{
	double angle;
	const char *axis;

	if( !PyArg_ParseTuple( args, "sd", &axis, &angle ) ) {
		return NULL;
	}

	if( strlen( axis ) != 1 ) {
		PyErr_SetString( PyExc_RuntimeError, "Illegal axis name" );
		return NULL;
	}

	int a = 0;
	int b = 0;

	double c = cos( angle*M_PI/180.0 );
	double s = sin( angle*M_PI/180.0 );

	if( axis[0] == 'Y' ) {
		s = -s;
	}

	Matrix4x4 ret;

	/* This is admittedly a little obfuscated.  Heh! */
	a = (axis[0]=='X')?1:0;
	b = (axis[0]=='Z')?1:2;
	ret[a][a] = c;
	ret[a][b] = -s;
	ret[b][a] = s;
	ret[b][b] = c;

	return (PyObject*)new_matrix( ret );
}

static PyObject *gr_new_translation_cmd( PyObject *self, PyObject *args )
{
	double xtrans;
	double ytrans;
	double ztrans;

	if( !PyArg_ParseTuple( args, "ddd", &xtrans, &ytrans, &ztrans ) ) {
		return NULL;
	}

	Matrix4x4 ret;
	ret[0][3] = xtrans;
	ret[1][3] = ytrans;
	ret[2][3] = ztrans;

	return (PyObject*)new_matrix( ret );
}

static PyObject *gr_new_scaling_cmd( PyObject *self, PyObject *args )
{
	double xscale;
	double yscale;
	double zscale;

	if( !PyArg_ParseTuple( args, "ddd", &xscale, &yscale, &zscale ) ) {
		return NULL;
	}

	Matrix4x4 ret;
	ret[0][0] = xscale;
	ret[1][1] = yscale;
	ret[2][2] = zscale;

	return (PyObject*)new_matrix( ret );
}

static PyObject *gr_transpose_cmd( PyObject *self, PyObject *args )
{
	PyMatrixObject *mat;

	if( !PyArg_ParseTuple( args, "O!", &Matrix_Type, &mat ) ) {
		return NULL;
	}

	Matrix4x4 a( mat->data );
	a = a.transpose();

	return (PyObject*)new_matrix( a );
}

static PyObject *gr_draw_model_cmd( PyObject *self, PyObject *args )
{
	if( !PyArg_ParseTuple( args, "" ) ) {
		return NULL;
	}

	gr_draw_model();

	Py_INCREF( Py_None );
	return Py_None;
}

PyObject *gr_load_model_cmd( PyObject *self, PyObject *args )
{
	const char *model_name;

	if( !PyArg_ParseTuple( args, "s", &model_name ) ) {
		return NULL;
	}

	gr_load_model( model_name );

	Py_INCREF( Py_None );
	return Py_None;
}

PyObject *gr_toggle_option_cmd( PyObject *self, PyObject *args )
{
	const char *opt;

	if( !PyArg_ParseTuple( args, "s", &opt ) ) {
		return NULL;
	}

	gr_toggle_option( opt );

	Py_INCREF( Py_None );
	return Py_None;
}

PyObject *gr_set_option_cmd( PyObject *self, PyObject *args )
{
	const char *opt;
	double o;

	if( !PyArg_ParseTuple( args, "sd", &opt, &o ) ) {
		return NULL;
	}

	gr_set_option( opt, o );

	Py_INCREF( Py_None );
	return Py_None;
}

PyObject *gr_do_reshape_cmd( PyObject *self, PyObject *args )
{
	if( !PyArg_ParseTuple( args, "" ) ) {
		return NULL;
	}

	gr_do_reshape();

	Py_INCREF( Py_None );
	return Py_None;
}

/*
 * Build a table containing all the methods that will be exported
 * by this module.
 */
static PyMethodDef gr_method_table[] = {
	{ "load_model", 		gr_load_model_cmd, 		METH_VARARGS, 	
		"Load a model" },
	{ "draw_model", 		gr_draw_model_cmd, 		METH_VARARGS, 	
		"Draw a model" },
	{ "do_reshape", 		gr_do_reshape_cmd, 		METH_VARARGS, 	
		"Reshape the viewport" },

	{ "identity",		gr_new_identity_cmd,	METH_VARARGS,
	  "Create an identity matrix" },
	{ "rotation",		gr_new_rotation_cmd,	METH_VARARGS,
	  "Create a rotation matrix" },
	{ "translation",	gr_new_translation_cmd,	METH_VARARGS,
	  "Create a translation matrix" },
	{ "scaling",			gr_new_scaling_cmd,	METH_VARARGS,
	  "Create a scaling matrix" },
	{ "transpose",			gr_transpose_cmd,	METH_VARARGS,
	  "Get the transpose" },

	{ "toggle",			gr_toggle_option_cmd,	METH_VARARGS,
	  "Toggle some option" },
	{ "set_option",			gr_set_option_cmd,	METH_VARARGS,
	  "Toggle some option" },

	{ NULL, 			NULL, 				0, NULL }
};

/* 
 * This function is called automatically by Python when this module
 * is first imported.  It adds the module and all the C functions into
 * the running interpreter.
 */
extern "C" {
	DL_EXPORT(void) initgr(void);
}

DL_EXPORT(void) initgr(void) 
{
	/* A little hack that comes from the Python documentation.
	   PyType_Type isn't defined statically, so we can only do this
	   assignment at run time.  Or something like that. */
	Matrix_Type.ob_type = &PyType_Type;

	/* A further hack that makes this work with C++ (there's a
	   problem with the usual method because of the 'staticforward'
	   declaration. */
	pMatrix_Type = &Matrix_Type;

	(void)Py_InitModule( "gr", gr_method_table );
}
