// Mersenne Chain Search
//
// Written by Alex Leong - adlleong@gmail.com
// Last updated October 2012
// 
// Using ECM factorization code by Alejandro Alpern
// Elliptic Curve Method (ECM) Prime Factorization
//
// Written by Dario Alejandro Alpern (Buenos Aires - Argentina)
// Last updated May 15th, 2011. See http://www.alpertron.com.ar/ECM.HTM
//
// Based in Yuji Kida's implementation for UBASIC interpreter
//
// No part of this code can be used for commercial purposes without
// the written consent from the author. Otherwise it can be used freely
// except that you have to write somewhere in the code this header.
//
// Useage:
// To find a Mersenne chain from 2^p-1 to a smaller number
// java MersenneSearch p
//
// Example:
// java MersenneSearch 5
// [31, 62, 61, 122, 121, 132, 99, 100, 75, 76, 57, 58, 29]

import java.util.*;
import java.util.concurrent.*;
import java.math.*;

public class MersenneSearch {

	static <T> T first(List<T> list){
		return list.get(0);
	}

	static <T> T last(List<T> list){
		return list.get(list.size()-1);
	}

  static BigInteger big1 = new BigInteger("1");
  static BigInteger big2 = new BigInteger("2");

  private ECMFactorizer ecm;
  private ExecutorService executor;

  public List<BigInteger> getOddDivisors(BigInteger n) throws Exception{
    ecm = new ECMFactorizer();
    executor = Executors.newSingleThreadExecutor();
    ecm.setNbrToFactor(n);
    //System.out.println("Executing...");
		executor.invokeAll(Arrays.asList(ecm), 5, TimeUnit.SECONDS);
    executor.shutdown();
    executor = null;
    List<BigInteger> divisors = ecm.getOddDivisors();
    ecm = null;
    System.gc();
    return divisors;
  }

  public static void main(String[] args) throws Exception{
    int exp = Integer.parseInt(args[0]);
    BigInteger start = big2.pow(exp).subtract(big1);
    MersenneSearch search = new MersenneSearch();
    search.search(start);
    System.exit(0);
  }

  public void search(BigInteger start) throws Exception{

    List<BigInteger> seq = new ArrayList<BigInteger>();
    seq.add(start);

    System.out.println(start);
    System.out.println();

		List<BigInteger> divisors1;
    List<BigInteger> divisors2;
		while( last(seq).compareTo(first(seq)) >= 0 ) {

      System.out.println();
      System.out.println("(Chain length "+seq.size()+")");
      System.out.println();

      divisors1 = getOddDivisors(last(seq));
			Collections.sort(divisors1);

loop: for( int i = 0; i < divisors1.size(); ++i ){
        BigInteger step = last(seq).add( divisors1.get(i) );
        if( ! seq.contains(step) ) {
          
          divisors2 = getOddDivisors(step);
          Collections.sort(divisors2);
          if( divisors2.size() > 1 ) {
          System.out.println("Smallest odd divisors are " + divisors2.get(0).toString() + ", " + divisors2.get(1).toString() );
          }else{
            System.out.println("Less than 2 odd divisors");
          }

          for( int j = divisors2.size()-1; j >= 0; --j ){
            BigInteger dest = step.subtract( divisors2.get(j) );
            if( ! seq.contains(dest) ) {

              System.out.println(" + "+divisors1.get(i).toString() + " (" + divisors1.get(i).toString().length() + ")" );
              System.out.println();
              System.out.println(step.toString() + " (" + step.toString().length() + ")" );
              System.out.println();
              System.out.println(" - "+divisors2.get(j).toString() + " (" + divisors2.get(j).toString().length() + ")" );
              System.out.println();
              System.out.println(dest.toString() + " (" + dest.toString().length() +")" );
              System.out.println();

              seq.add(step);
              seq.add(dest);
              break loop;
            }
          }
        }
      }

	System.gc();
		}

    System.out.println();
    System.out.println(seq.toString());
						
	}

}
